<?php
session_start();

// Verificar si el usuario está logueado
if (!isset($_SESSION['user_id'])) {
    header("Location: index.php");
    exit();
}

require_once "config/database_advanced.php";
require_once "includes/log_viewer.php";

// Inicializar el gestor de base de datos avanzado
$dbManager = AdvancedDatabaseManager::getInstance();

// Obtener estadísticas actuales
$stats = $dbManager->getAdvancedStats();
$health = $dbManager->checkConnectionHealth();

// Inicializar el visualizador de logs
$logViewer = new LogViewer();
$logs = $logViewer->loadLogs();

// Aplicar filtros si se proporcionan
$filters = [];
if (!empty($_GET['type'])) $filters['type'] = $_GET['type'];
if (!empty($_GET['action'])) $filters['action'] = $_GET['action'];
if (!empty($_GET['date'])) $filters['date'] = $_GET['date'];
if (!empty($_GET['search'])) $filters['search'] = $_GET['search'];

$filteredLogs = $logViewer->filterLogs($filters);
$logStats = $logViewer->getLogStats();

// Analizar logs para detectar conexiones abiertas
$openedConnections = 0;
$closedConnections = 0;
$openConnections = [];

foreach ($logs as $log) {
    if ($log['action'] === 'OPENING') {
        $openedConnections++;
        $openConnections[] = [
            'type' => $log['type'],
            'timestamp' => $log['timestamp'],
            'message' => $log['message']
        ];
    } elseif ($log['action'] === 'CLOSING') {
        $closedConnections++;
    }
}

// Probar conexiones activas
$activeConnections = [];
try {
    $crmConnection = $dbManager->getCRMConnection();
    $activeConnections['crm'] = true;
} catch (Exception $e) {
    $activeConnections['crm'] = false;
}

try {
    $jadeConnection = $dbManager->getJadeConnection();
    $activeConnections['jade'] = true;
} catch (Exception $e) {
    $activeConnections['jade'] = false;
}

// Obtener información de memoria
$memoryUsage = memory_get_usage(true);
$memoryPeak = memory_get_peak_usage(true);
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Verificador de Conexiones - Panel de Estadísticas</title>
    <link rel="stylesheet" href="assets/css/dark-theme.css">
    <link rel="stylesheet" href="assets/css/monitoring.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
</head>
<body>
    <div class="monitoring-container">
        <!-- Header -->
        <div class="monitoring-header">
            <h1 class="monitoring-title">
                <i class="fas fa-search"></i> Verificador de Conexiones Abiertas
            </h1>
            <p class="monitoring-subtitle">
                Análisis detallado de conexiones de base de datos y logs del sistema
            </p>
            <div style="text-align: center; margin-top: 15px;">
                <a href="herramientas_monitoreo.php" class="btn-monitoring btn-primary">
                    <i class="fas fa-arrow-left"></i> Volver a Herramientas
                </a>
            </div>
        </div>

        <!-- Estado Actual -->
        <div class="monitoring-grid">
            <div class="monitoring-card">
                <div class="card-header">
                    <div class="card-icon" style="background: linear-gradient(45deg, #667eea, #764ba2);">
                        <i class="fas fa-database"></i>
                    </div>
                    <h3 class="card-title">Estado Actual</h3>
                </div>
                
                <div style="margin-bottom: 20px;">
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Conexión CRM:</strong></span>
                        <span class="status-indicator <?php echo $activeConnections['crm'] ? 'status-active' : 'status-inactive'; ?>">
                            <?php echo $activeConnections['crm'] ? '🔴 ABIERTA' : '✅ CERRADA'; ?>
                        </span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Conexión Jade:</strong></span>
                        <span class="status-indicator <?php echo $activeConnections['jade'] ? 'status-active' : 'status-inactive'; ?>">
                            <?php echo $activeConnections['jade'] ? '🔴 ABIERTA' : '✅ CERRADA'; ?>
                        </span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Total Conexiones:</strong></span>
                        <span class="status-indicator status-active">
                            <?php echo array_sum($activeConnections); ?>
                        </span>
                    </div>
                </div>
            </div>

            <div class="monitoring-card">
                <div class="card-header">
                    <div class="card-icon" style="background: linear-gradient(45deg, #00b894, #00cec9);">
                        <i class="fas fa-chart-bar"></i>
                    </div>
                    <h3 class="card-title">Análisis de Logs</h3>
                </div>
                
                <div style="margin-bottom: 20px;">
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Conexiones Abiertas:</strong></span>
                        <span class="status-indicator status-warning">
                            <?php echo $openedConnections; ?>
                        </span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Conexiones Cerradas:</strong></span>
                        <span class="status-indicator status-active">
                            <?php echo $closedConnections; ?>
                        </span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Diferencia:</strong></span>
                        <span class="status-indicator <?php echo ($openedConnections - $closedConnections) > 0 ? 'status-inactive' : 'status-active'; ?>">
                            <?php echo $openedConnections - $closedConnections; ?>
                        </span>
                    </div>
                </div>
            </div>

            <div class="monitoring-card">
                <div class="card-header">
                    <div class="card-icon" style="background: linear-gradient(45deg, #fdcb6e, #e17055);">
                        <i class="fas fa-memory"></i>
                    </div>
                    <h3 class="card-title">Uso de Memoria</h3>
                </div>
                
                <div style="margin-bottom: 20px;">
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Memoria Actual:</strong></span>
                        <span><?php echo number_format($memoryUsage / 1024 / 1024, 2); ?> MB</span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Memoria Pico:</strong></span>
                        <span><?php echo number_format($memoryPeak / 1024 / 1024, 2); ?> MB</span>
                    </div>
                </div>
            </div>

            <div class="monitoring-card">
                <div class="card-header">
                    <div class="card-icon" style="background: linear-gradient(45deg, #e17055, #d63031);">
                        <i class="fas fa-cogs"></i>
                    </div>
                    <h3 class="card-title">Acciones</h3>
                </div>
                
                <div style="margin-bottom: 20px;">
                    <p style="color: #7f8c8d; font-size: 0.9rem; margin-bottom: 15px;">
                        Acciones para gestionar las conexiones detectadas
                    </p>
                </div>
                
                <div style="display: flex; flex-direction: column; gap: 10px;">
                    <a href="cerrar_conexiones.php" class="btn-monitoring btn-danger" target="_blank">
                        <i class="fas fa-times-circle"></i> Cerrar Todas las Conexiones
                    </a>
                    
                    <button class="btn-monitoring btn-success" onclick="location.reload()">
                        <i class="fas fa-sync-alt"></i> Actualizar Verificación
                    </button>
                </div>
            </div>
        </div>

        <!-- Últimas Conexiones Abiertas -->
        <?php if (!empty($openConnections)): ?>
        <div class="monitoring-card">
            <div class="card-header">
                <div class="card-icon" style="background: linear-gradient(45deg, #e17055, #d63031);">
                    <i class="fas fa-exclamation-triangle"></i>
                </div>
                <h3 class="card-title">Últimas Conexiones Abiertas</h3>
            </div>
            
            <div style="overflow-x: auto;">
                <table class="logs-table">
                    <thead>
                        <tr>
                            <th>Timestamp</th>
                            <th>Tipo</th>
                            <th>Mensaje</th>
                            <th>Estado</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach (array_slice($openConnections, 0, 10) as $connection): ?>
                        <tr>
                            <td>
                                <strong><?php echo date('H:i:s', strtotime($connection['timestamp'])); ?></strong><br>
                                <small><?php echo date('d/m/Y', strtotime($connection['timestamp'])); ?></small>
                            </td>
                            <td>
                                <span class="status-indicator <?php echo strtolower($connection['type']) === 'crm' ? 'status-active' : 'status-warning'; ?>">
                                    <?php echo $connection['type']; ?>
                                </span>
                            </td>
                            <td><?php echo htmlspecialchars($connection['message']); ?></td>
                            <td>
                                <span class="status-indicator status-inactive">
                                    🔴 ABIERTA
                                </span>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        <?php endif; ?>

        <!-- Visualizador de Logs Mejorado -->
        <?php $logViewer->renderLogsTable($filteredLogs); ?>
    </div>

    <script>
    // Auto-refresh cada 30 segundos
    setInterval(function() {
        location.reload();
    }, 30000);
    </script>
</body>
</html>
