<?php
session_start();

// Verificar si el usuario está logueado
if (!isset($_SESSION['user_id'])) {
    header("Location: index.php");
    exit();
}

require_once "config/database_advanced.php";
require_once "includes/log_viewer.php";

// Inicializar el gestor de base de datos avanzado
$dbManager = AdvancedDatabaseManager::getInstance();

// Obtener estadísticas actuales
$stats = $dbManager->getAdvancedStats();
$health = $dbManager->checkConnectionHealth();

// Inicializar el visualizador de logs
$logViewer = new LogViewer();
$logs = $logViewer->loadLogs();

// Aplicar filtros si se proporcionan
$filters = [];
if (!empty($_GET['type'])) $filters['type'] = $_GET['type'];
if (!empty($_GET['action'])) $filters['action'] = $_GET['action'];
if (!empty($_GET['date'])) $filters['date'] = $_GET['date'];
if (!empty($_GET['search'])) $filters['search'] = $_GET['search'];

$filteredLogs = $logViewer->filterLogs($filters);
$logStats = $logViewer->getLogStats();

// Probar conexiones
$testResults = [];
try {
    $start = microtime(true);
    $crmConnection = $dbManager->getCRMConnection();
    $crmTime = (microtime(true) - $start) * 1000;
    $testResults['crm'] = ['success' => true, 'time' => $crmTime];
} catch (Exception $e) {
    $testResults['crm'] = ['success' => false, 'error' => $e->getMessage()];
}

try {
    $start = microtime(true);
    $jadeConnection = $dbManager->getJadeConnection();
    $jadeTime = (microtime(true) - $start) * 1000;
    $testResults['jade'] = ['success' => true, 'time' => $jadeTime];
} catch (Exception $e) {
    $testResults['jade'] = ['success' => false, 'error' => $e->getMessage()];
}

// Verificar singleton
$crm1 = $dbManager->getCRMConnection();
$crm2 = $dbManager->getCRMConnection();
$jade1 = $dbManager->getJadeConnection();
$jade2 = $dbManager->getJadeConnection();

$singletonCRM = ($crm1 === $crm2);
$singletonJade = ($jade1 === $jade2);

// Limpiar conexiones inactivas
$cleanupResult = $dbManager->cleanupInactiveConnections(300); // 5 minutos
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Monitor Avanzado - Panel de Estadísticas</title>
    <link rel="stylesheet" href="assets/css/dark-theme.css">
    <link rel="stylesheet" href="assets/css/monitoring.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
</head>
<body>
    <div class="monitoring-container">
        <!-- Header -->
        <div class="monitoring-header">
            <h1 class="monitoring-title">
                <i class="fas fa-chart-line"></i> Monitor Avanzado de Conexiones
            </h1>
            <p class="monitoring-subtitle">
                Sistema avanzado de monitoreo y gestión de conexiones de base de datos
            </p>
            <div style="text-align: center; margin-top: 15px;">
                <a href="herramientas_monitoreo.php" class="btn-monitoring btn-primary">
                    <i class="fas fa-arrow-left"></i> Volver a Herramientas
                </a>
            </div>
        </div>

        <!-- Estado del Sistema -->
        <div class="monitoring-grid">
            <div class="monitoring-card">
                <div class="card-header">
                    <div class="card-icon" style="background: linear-gradient(45deg, #667eea, #764ba2);">
                        <i class="fas fa-database"></i>
                    </div>
                    <h3 class="card-title">Estado de Conexiones</h3>
                </div>
                
                <div style="margin-bottom: 20px;">
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Conexión CRM:</strong></span>
                        <span class="status-indicator <?php echo $stats['crm_connected'] ? 'status-active' : 'status-inactive'; ?>">
                            <?php echo $stats['crm_connected'] ? '✅ Activa' : '❌ Inactiva'; ?>
                        </span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Conexión Jade:</strong></span>
                        <span class="status-indicator <?php echo $stats['jade_connected'] ? 'status-active' : 'status-inactive'; ?>">
                            <?php echo $stats['jade_connected'] ? '✅ Activa' : '❌ Inactiva'; ?>
                        </span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Total Conexiones:</strong></span>
                        <span class="status-indicator status-active">
                            <?php echo $stats['total_connections']; ?>/<?php echo $stats['max_connections']; ?>
                        </span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Timeout:</strong></span>
                        <span><?php echo $stats['connection_timeout']; ?>s</span>
                    </div>
                </div>
            </div>

            <div class="monitoring-card">
                <div class="card-header">
                    <div class="card-icon" style="background: linear-gradient(45deg, #00b894, #00cec9);">
                        <i class="fas fa-heartbeat"></i>
                    </div>
                    <h3 class="card-title">Health Check</h3>
                </div>
                
                <div style="margin-bottom: 20px;">
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>CRM:</strong></span>
                        <span class="status-indicator <?php echo $health['crm']['status'] === 'healthy' ? 'status-active' : 'status-warning'; ?>">
                            <?php echo strtoupper($health['crm']['status']); ?>
                        </span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Jade:</strong></span>
                        <span class="status-indicator <?php echo $health['jade']['status'] === 'healthy' ? 'status-active' : 'status-warning'; ?>">
                            <?php echo strtoupper($health['jade']['status']); ?>
                        </span>
                    </div>
                </div>
            </div>

            <div class="monitoring-card">
                <div class="card-header">
                    <div class="card-icon" style="background: linear-gradient(45deg, #fdcb6e, #e17055);">
                        <i class="fas fa-tachometer-alt"></i>
                    </div>
                    <h3 class="card-title">Métricas de Rendimiento</h3>
                </div>
                
                <div style="margin-bottom: 20px;">
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Tiempo Promedio:</strong></span>
                        <span><?php 
                            $avgTime = 0;
                            if (isset($stats['performance_summary']['CRM']['connection_time']['avg'])) {
                                $avgTime = $stats['performance_summary']['CRM']['connection_time']['avg'] * 1000;
                            }
                            echo number_format($avgTime, 2); 
                        ?>ms</span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Conexiones Reutilizadas:</strong></span>
                        <span><?php echo count($stats['recent_logs']); ?></span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Memoria Usada:</strong></span>
                        <span><?php echo number_format(memory_get_usage(true) / 1024 / 1024, 2); ?> MB</span>
                    </div>
                </div>
            </div>

            <div class="monitoring-card">
                <div class="card-header">
                    <div class="card-icon" style="background: linear-gradient(45deg, #6c5ce7, #a29bfe);">
                        <i class="fas fa-shield-alt"></i>
                    </div>
                    <h3 class="card-title">Verificación Singleton</h3>
                </div>
                
                <div style="margin-bottom: 20px;">
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>CRM Singleton:</strong></span>
                        <span class="status-indicator <?php echo $singletonCRM ? 'status-active' : 'status-inactive'; ?>">
                            <?php echo $singletonCRM ? '✅ Funcionando' : '❌ Error'; ?>
                        </span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Jade Singleton:</strong></span>
                        <span class="status-indicator <?php echo $singletonJade ? 'status-active' : 'status-inactive'; ?>">
                            <?php echo $singletonJade ? '✅ Funcionando' : '❌ Error'; ?>
                        </span>
                    </div>
                </div>
            </div>
        </div>

        <!-- Pruebas de Conexión -->
        <div class="monitoring-card">
            <div class="card-header">
                <div class="card-icon" style="background: linear-gradient(45deg, #e17055, #d63031);">
                    <i class="fas fa-vial"></i>
                </div>
                <h3 class="card-title">Pruebas de Conexión</h3>
            </div>
            
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 20px;">
                <div>
                    <h5>CRM Database</h5>
                    <?php if ($testResults['crm']['success']): ?>
                        <div class="status-indicator status-active">
                            ✅ Conexión exitosa en <?php echo number_format($testResults['crm']['time'], 2); ?>ms
                        </div>
                    <?php else: ?>
                        <div class="status-indicator status-inactive">
                            ❌ Error: <?php echo htmlspecialchars($testResults['crm']['error']); ?>
                        </div>
                    <?php endif; ?>
                </div>
                
                <div>
                    <h5>Jade Database</h5>
                    <?php if ($testResults['jade']['success']): ?>
                        <div class="status-indicator status-active">
                            ✅ Conexión exitosa en <?php echo number_format($testResults['jade']['time'], 2); ?>ms
                        </div>
                    <?php else: ?>
                        <div class="status-indicator status-inactive">
                            ❌ Error: <?php echo htmlspecialchars($testResults['jade']['error']); ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- Limpieza de Conexiones -->
        <div class="monitoring-card">
            <div class="card-header">
                <div class="card-icon" style="background: linear-gradient(45deg, #00b894, #00cec9);">
                    <i class="fas fa-broom"></i>
                </div>
                <h3 class="card-title">Limpieza de Conexiones</h3>
            </div>
            
            <div style="margin-bottom: 20px;">
                <p>Conexiones inactivas limpiadas: <strong><?php echo $cleanupResult; ?></strong></p>
                <p>Se eliminaron conexiones inactivas por más de 5 minutos.</p>
            </div>
            
            <div style="text-align: center;">
                <button class="btn-monitoring btn-success" onclick="location.reload()">
                    <i class="fas fa-sync-alt"></i> Actualizar
                </button>
            </div>
        </div>

        <!-- Visualizador de Logs Mejorado -->
        <?php $logViewer->renderLogsTable($filteredLogs); ?>
    </div>

    <script>
    // Auto-refresh cada 30 segundos
    setInterval(function() {
        location.reload();
    }, 30000);
    </script>
</body>
</html>
