<?php
/**
 * Script de Limpieza de Emergencia
 * Cierra todas las conexiones acumuladas y resetea el sistema
 */

session_start();

// Verificar si el usuario está logueado
if (!isset($_SESSION['user_id'])) {
    header("Location: index.php");
    exit();
}

require_once "config/database_advanced.php";

// Inicializar el gestor de base de datos avanzado
$dbManager = AdvancedDatabaseManager::getInstance();

// Obtener estadísticas antes de la limpieza
$statsBefore = $dbManager->getAdvancedStats();

// Realizar limpieza de emergencia
$cleanupResult = $dbManager->cleanupInactiveConnections(0); // Forzar limpieza inmediata

// Cerrar todas las conexiones explícitamente
$dbManager->closeConnections();

// Limpiar archivo de logs para resetear contadores
$logFile = 'logs/database_' . date('Y-m-d') . '.log';
if (file_exists($logFile)) {
    // Crear backup del log actual
    $backupFile = 'logs/database_' . date('Y-m-d') . '_backup_' . date('H-i-s') . '.log';
    copy($logFile, $backupFile);
    
    // Limpiar el log actual
    file_put_contents($logFile, '');
}

// Obtener estadísticas después de la limpieza
$statsAfter = $dbManager->getAdvancedStats();

// Log de la limpieza de emergencia
$emergencyLog = [
    'timestamp' => date('Y-m-d H:i:s'),
    'type' => 'SYSTEM',
    'action' => 'EMERGENCY_CLEANUP',
    'message' => 'Limpieza de emergencia ejecutada - Conexiones cerradas: ' . $cleanupResult,
    'memory_usage' => memory_get_usage(true),
    'peak_memory' => memory_get_peak_usage(true)
];

$emergencyLogFile = 'logs/emergency_' . date('Y-m-d') . '.log';
file_put_contents($emergencyLogFile, json_encode($emergencyLog) . "\n", FILE_APPEND | LOCK_EX);

// Obtener información del sistema
$systemInfo = [
    'php_version' => PHP_VERSION,
    'memory_limit' => ini_get('memory_limit'),
    'max_execution_time' => ini_get('max_execution_time'),
    'server_time' => date('Y-m-d H:i:s'),
    'uptime' => function_exists('sys_getloadavg') ? sys_getloadavg() : 'N/A'
];

?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Limpieza de Emergencia - Panel de Estadísticas</title>
    <link rel="stylesheet" href="assets/css/dark-theme.css">
    <link rel="stylesheet" href="assets/css/monitoring.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
</head>
<body>
    <div class="monitoring-container">
        <!-- Header -->
        <div class="monitoring-header">
            <h1 class="monitoring-title">
                <i class="fas fa-fire-extinguisher"></i> Limpieza de Emergencia Completada
            </h1>
            <p class="monitoring-subtitle">
                Todas las conexiones acumuladas han sido cerradas y el sistema ha sido reseteado
            </p>
            <div style="text-align: center; margin-top: 15px;">
                <a href="diagnostico_conexiones_detallado.php" class="btn-monitoring btn-primary">
                    <i class="fas fa-search"></i> Verificar Estado
                </a>
                <a href="herramientas_monitoreo.php" class="btn-monitoring btn-success">
                    <i class="fas fa-tools"></i> Volver a Herramientas
                </a>
            </div>
        </div>

        <!-- Resultado de la Limpieza -->
        <div class="monitoring-card" style="border: 2px solid #27ae60; background: linear-gradient(45deg, #27ae60, #2ecc71);">
            <div class="card-header">
                <div class="card-icon" style="background: linear-gradient(45deg, #27ae60, #2ecc71);">
                    <i class="fas fa-check-circle"></i>
                </div>
                <h3 class="card-title" style="color: white;">LIMPIEZA EXITOSA</h3>
            </div>
            
            <div style="margin-bottom: 20px; color: white;">
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                    <span><strong>Conexiones Limpiadas:</strong></span>
                    <span style="font-size: 1.2em; font-weight: bold;"><?php echo $cleanupResult; ?></span>
                </div>
                
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                    <span><strong>Estado Actual:</strong></span>
                    <span style="font-size: 1.2em; font-weight: bold; color: #f1c40f;">SISTEMA LIMPIO</span>
                </div>
                
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                    <span><strong>Memoria Liberada:</strong></span>
                    <span style="font-size: 1.2em; font-weight: bold;"><?php echo number_format(memory_get_usage(true) / 1024 / 1024, 2); ?> MB</span>
                </div>
            </div>
        </div>

        <!-- Comparación Antes vs Después -->
        <div class="monitoring-grid">
            <div class="monitoring-card">
                <div class="card-header">
                    <div class="card-icon" style="background: linear-gradient(45deg, #e74c3c, #c0392b);">
                        <i class="fas fa-exclamation-triangle"></i>
                    </div>
                    <h3 class="card-title">Estado ANTES</h3>
                </div>
                
                <div style="margin-bottom: 20px;">
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Conexión CRM:</strong></span>
                        <span class="status-indicator <?php echo $statsBefore['crm_connected'] ? 'status-active' : 'status-inactive'; ?>">
                            <?php echo $statsBefore['crm_connected'] ? '🔴 ABIERTA' : '✅ CERRADA'; ?>
                        </span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Conexión Jade:</strong></span>
                        <span class="status-indicator <?php echo $statsBefore['jade_connected'] ? 'status-active' : 'status-inactive'; ?>">
                            <?php echo $statsBefore['jade_connected'] ? '🔴 ABIERTA' : '✅ CERRADA'; ?>
                        </span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Total Conexiones:</strong></span>
                        <span class="status-indicator status-warning">
                            <?php echo $statsBefore['total_connections']; ?>
                        </span>
                    </div>
                </div>
            </div>

            <div class="monitoring-card">
                <div class="card-header">
                    <div class="card-icon" style="background: linear-gradient(45deg, #27ae60, #2ecc71);">
                        <i class="fas fa-check-circle"></i>
                    </div>
                    <h3 class="card-title">Estado DESPUÉS</h3>
                </div>
                
                <div style="margin-bottom: 20px;">
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Conexión CRM:</strong></span>
                        <span class="status-indicator <?php echo $statsAfter['crm_connected'] ? 'status-active' : 'status-inactive'; ?>">
                            <?php echo $statsAfter['crm_connected'] ? '🔴 ABIERTA' : '✅ CERRADA'; ?>
                        </span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Conexión Jade:</strong></span>
                        <span class="status-indicator <?php echo $statsAfter['jade_connected'] ? 'status-active' : 'status-inactive'; ?>">
                            <?php echo $statsAfter['jade_connected'] ? '🔴 ABIERTA' : '✅ CERRADA'; ?>
                        </span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Total Conexiones:</strong></span>
                        <span class="status-indicator status-active">
                            <?php echo $statsAfter['total_connections']; ?>
                        </span>
                    </div>
                </div>
            </div>

            <div class="monitoring-card">
                <div class="card-header">
                    <div class="card-icon" style="background: linear-gradient(45deg, #fdcb6e, #e17055);">
                        <i class="fas fa-memory"></i>
                    </div>
                    <h3 class="card-title">Memoria</h3>
                </div>
                
                <div style="margin-bottom: 20px;">
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Memoria Actual:</strong></span>
                        <span><?php echo number_format(memory_get_usage(true) / 1024 / 1024, 2); ?> MB</span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Memoria Pico:</strong></span>
                        <span><?php echo number_format(memory_get_peak_usage(true) / 1024 / 1024, 2); ?> MB</span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Límite de Memoria:</strong></span>
                        <span><?php echo $systemInfo['memory_limit']; ?></span>
                    </div>
                </div>
            </div>

            <div class="monitoring-card">
                <div class="card-header">
                    <div class="card-icon" style="background: linear-gradient(45deg, #6c5ce7, #a29bfe);">
                        <i class="fas fa-info-circle"></i>
                    </div>
                    <h3 class="card-title">Información</h3>
                </div>
                
                <div style="margin-bottom: 20px;">
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Backup Creado:</strong></span>
                        <span class="status-indicator status-active">✅ Sí</span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Log de Emergencia:</strong></span>
                        <span class="status-indicator status-active">✅ Creado</span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Hora de Limpieza:</strong></span>
                        <span><?php echo date('H:i:s'); ?></span>
                    </div>
                </div>
            </div>
        </div>

        <!-- Acciones Post-Limpieza -->
        <div class="monitoring-card">
            <div class="card-header">
                <div class="card-icon" style="background: linear-gradient(45deg, #e17055, #d63031);">
                    <i class="fas fa-cogs"></i>
                </div>
                <h3 class="card-title">Acciones Recomendadas</h3>
            </div>
            
            <div style="margin-bottom: 20px;">
                <p style="color: #7f8c8d; font-size: 0.9rem; margin-bottom: 15px;">
                    Después de la limpieza de emergencia, se recomienda:
                </p>
                
                <ul style="list-style: none; padding: 0; color: #7f8c8d;">
                    <li style="margin-bottom: 8px;">✅ Verificar que el sistema funcione correctamente</li>
                    <li style="margin-bottom: 8px;">✅ Monitorear las conexiones durante las próximas horas</li>
                    <li style="margin-bottom: 8px;">✅ Implementar la corrección del sistema para prevenir futuros problemas</li>
                    <li style="margin-bottom: 8px;">✅ Revisar los logs de emergencia si es necesario</li>
                </ul>
            </div>
            
            <div style="display: flex; flex-direction: column; gap: 10px;">
                <a href="correccion_conexiones.php" class="btn-monitoring btn-warning">
                    <i class="fas fa-wrench"></i> Implementar Corrección del Sistema
                </a>
                
                <a href="diagnostico_conexiones_detallado.php" class="btn-monitoring btn-primary">
                    <i class="fas fa-search"></i> Verificar Estado Actual
                </a>
                
                <a href="herramientas_monitoreo.php" class="btn-monitoring btn-success">
                    <i class="fas fa-tools"></i> Volver a Herramientas
                </a>
            </div>
        </div>

        <!-- Información del Sistema -->
        <div class="monitoring-card">
            <div class="card-header">
                <div class="card-icon" style="background: linear-gradient(45deg, #6c5ce7, #a29bfe);">
                    <i class="fas fa-info-circle"></i>
                </div>
                <h3 class="card-title">Información del Sistema</h3>
            </div>
            
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px;">
                <div>
                    <strong>PHP Version:</strong> <?php echo $systemInfo['php_version']; ?>
                </div>
                <div>
                    <strong>Memory Limit:</strong> <?php echo $systemInfo['memory_limit']; ?>
                </div>
                <div>
                    <strong>Max Execution Time:</strong> <?php echo $systemInfo['max_execution_time']; ?>s
                </div>
                <div>
                    <strong>Server Time:</strong> <?php echo $systemInfo['server_time']; ?>
                </div>
                <?php if ($systemInfo['uptime'] !== 'N/A'): ?>
                <div>
                    <strong>System Load:</strong> <?php echo implode(', ', array_map('number_format', $systemInfo['uptime'])); ?>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <script>
    // Auto-redirect después de 30 segundos
    setTimeout(function() {
        window.location.href = 'diagnostico_conexiones_detallado.php';
    }, 30000);
    </script>
</body>
</html>
