<?php
/**
 * Componente para visualización de logs mejorada
 * Incluye filtros, búsqueda y tabla organizada
 */

class LogViewer {
    private $logFile;
    private $logs = [];
    
    public function __construct($logFile = null) {
        $this->logFile = $logFile ?: 'logs/database_' . date('Y-m-d') . '.log';
    }
    
    /**
     * Cargar logs desde archivo
     */
    public function loadLogs() {
        if (!file_exists($this->logFile)) {
            return [];
        }
        
        $lines = file($this->logFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        $this->logs = [];
        
        foreach ($lines as $line) {
            $logEntry = $this->parseLogLine($line);
            if ($logEntry) {
                $this->logs[] = $logEntry;
            }
        }
        
        // Ordenar por timestamp descendente (más recientes primero)
        usort($this->logs, function($a, $b) {
            return strtotime($b['timestamp']) - strtotime($a['timestamp']);
        });
        
        return $this->logs;
    }
    
    /**
     * Parsear línea de log
     */
    private function parseLogLine($line) {
        // Formato esperado: {"timestamp":"2025-08-25 14:13:13","type":"CRM","action":"OPENING","message":"Iniciando conexión CRM"}
        $logData = json_decode($line, true);
        if ($logData && isset($logData['timestamp'], $logData['type'], $logData['action'], $logData['message'])) {
            // Determinar icono basado en la acción
            $icon = '📝'; // default
            switch($logData['action']) {
                case 'OPENING':
                    $icon = '🔓';
                    break;
                case 'SUCCESS':
                    $icon = '✅';
                    break;
                case 'REUSE':
                    $icon = '🔄';
                    break;
                case 'CLOSING':
                    $icon = '🔒';
                    break;
                case 'ERROR':
                    $icon = '❌';
                    break;
                case 'CLEANUP':
                    $icon = '🧹';
                    break;
            }
            
            return [
                'icon' => $icon,
                'timestamp' => $logData['timestamp'],
                'type' => $logData['type'],
                'action' => $logData['action'],
                'message' => $logData['message'],
                'raw' => $icon . ' [' . $logData['timestamp'] . '] ' . $logData['type'] . ' - ' . $logData['action'] . ': ' . $logData['message']
            ];
        }
        return null;
    }
    
    /**
     * Filtrar logs
     */
    public function filterLogs($filters = []) {
        $filtered = $this->logs;
        
        // Filtro por tipo
        if (!empty($filters['type']) && $filters['type'] !== 'all') {
            $filtered = array_filter($filtered, function($log) use ($filters) {
                return strtolower($log['type']) === strtolower($filters['type']);
            });
        }
        
        // Filtro por acción
        if (!empty($filters['action']) && $filters['action'] !== 'all') {
            $filtered = array_filter($filtered, function($log) use ($filters) {
                return strtolower($log['action']) === strtolower($filters['action']);
            });
        }
        
        // Filtro por fecha
        if (!empty($filters['date'])) {
            $filtered = array_filter($filtered, function($log) use ($filters) {
                return date('Y-m-d', strtotime($log['timestamp'])) === $filters['date'];
            });
        }
        
        // Filtro por búsqueda
        if (!empty($filters['search'])) {
            $search = strtolower($filters['search']);
            $filtered = array_filter($filtered, function($log) use ($search) {
                return strpos(strtolower($log['message']), $search) !== false ||
                       strpos(strtolower($log['type']), $search) !== false ||
                       strpos(strtolower($log['action']), $search) !== false;
            });
        }
        
        return array_values($filtered);
    }
    
    /**
     * Obtener estadísticas de logs
     */
    public function getLogStats() {
        $stats = [
            'total' => count($this->logs),
            'types' => [],
            'actions' => [],
            'today' => 0,
            'errors' => 0
        ];
        
        foreach ($this->logs as $log) {
            // Contar por tipo
            $stats['types'][$log['type']] = ($stats['types'][$log['type']] ?? 0) + 1;
            
            // Contar por acción
            $stats['actions'][$log['action']] = ($stats['actions'][$log['action']] ?? 0) + 1;
            
            // Contar de hoy
            if (date('Y-m-d', strtotime($log['timestamp'])) === date('Y-m-d')) {
                $stats['today']++;
            }
            
            // Contar errores
            if ($log['action'] === 'ERROR') {
                $stats['errors']++;
            }
        }
        
        return $stats;
    }
    
    /**
     * Renderizar tabla de logs
     */
    public function renderLogsTable($logs = null, $showFilters = true) {
        if ($logs === null) {
            $logs = $this->logs;
        }
        
        $stats = $this->getLogStats();
        
        ?>
        <div class="logs-container">
            <?php if ($showFilters): ?>
            <div class="logs-controls">
                <div class="filter-group">
                    <label class="filter-label">Tipo:</label>
                    <select class="filter-select" id="filterType">
                        <option value="all">Todos</option>
                        <option value="crm">CRM</option>
                        <option value="jade">Jade</option>
                    </select>
                </div>
                
                <div class="filter-group">
                    <label class="filter-label">Acción:</label>
                    <select class="filter-select" id="filterAction">
                        <option value="all">Todas</option>
                        <option value="opening">Apertura</option>
                        <option value="success">Éxito</option>
                        <option value="error">Error</option>
                        <option value="reuse">Reutilización</option>
                        <option value="closing">Cierre</option>
                        <option value="cleanup">Limpieza</option>
                    </select>
                </div>
                
                <div class="filter-group">
                    <label class="filter-label">Fecha:</label>
                    <input type="date" class="filter-input" id="filterDate" value="<?php echo date('Y-m-d'); ?>">
                </div>
                
                <div class="filter-group search-box">
                    <label class="filter-label">Buscar:</label>
                    <input type="text" class="filter-input" id="searchLogs" placeholder="Buscar en logs...">
                </div>
                
                <button class="btn-monitoring btn-primary" onclick="applyFilters()">
                    🔍 Aplicar Filtros
                </button>
                
                <button class="btn-monitoring btn-success" onclick="exportLogs()">
                    📥 Exportar
                </button>
            </div>
            <?php endif; ?>
            
            <!-- Estadísticas -->
            <div class="metrics-grid">
                <div class="metric-card">
                    <div class="metric-value"><?php echo $stats['total']; ?></div>
                    <div class="metric-label">Total de Logs</div>
                </div>
                <div class="metric-card">
                    <div class="metric-value"><?php echo $stats['today']; ?></div>
                    <div class="metric-label">Hoy</div>
                </div>
                <div class="metric-card">
                    <div class="metric-value"><?php echo $stats['errors']; ?></div>
                    <div class="metric-label">Errores</div>
                </div>
                <div class="metric-card">
                    <div class="metric-value"><?php echo count($logs); ?></div>
                    <div class="metric-label">Filtrados</div>
                </div>
            </div>
            
            <!-- Tabla de logs -->
            <div style="overflow-x: auto;">
                <table class="logs-table" id="logsTable">
                    <thead>
                        <tr>
                            <th>Timestamp</th>
                            <th>Tipo</th>
                            <th>Acción</th>
                            <th>Mensaje</th>
                            <th>Estado</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($logs)): ?>
                        <tr>
                            <td colspan="5" style="text-align: center; padding: 40px; color: #7f8c8d;">
                                No se encontraron logs para mostrar
                            </td>
                        </tr>
                        <?php else: ?>
                            <?php foreach ($logs as $log): ?>
                            <tr>
                                <td>
                                    <strong><?php echo date('H:i:s', strtotime($log['timestamp'])); ?></strong><br>
                                    <small><?php echo date('d/m/Y', strtotime($log['timestamp'])); ?></small>
                                </td>
                                <td>
                                    <span class="status-indicator <?php echo strtolower($log['type']) === 'crm' ? 'status-active' : 'status-warning'; ?>">
                                        <?php echo $log['type']; ?>
                                    </span>
                                </td>
                                <td>
                                    <span class="log-action log-<?php echo strtolower($log['action']); ?>">
                                        <?php echo $log['icon']; ?> <?php echo $log['action']; ?>
                                    </span>
                                </td>
                                <td><?php echo htmlspecialchars($log['message']); ?></td>
                                <td>
                                    <?php
                                    $statusClass = 'status-active';
                                    $statusText = 'OK';
                                    
                                    if ($log['action'] === 'ERROR') {
                                        $statusClass = 'status-inactive';
                                        $statusText = 'Error';
                                    } elseif ($log['action'] === 'OPENING') {
                                        $statusClass = 'status-warning';
                                        $statusText = 'Abriendo';
                                    }
                                    ?>
                                    <span class="status-indicator <?php echo $statusClass; ?>">
                                        <?php echo $statusText; ?>
                                    </span>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
        
        <script>
        function applyFilters() {
            const type = document.getElementById('filterType').value;
            const action = document.getElementById('filterAction').value;
            const date = document.getElementById('filterDate').value;
            const search = document.getElementById('searchLogs').value;
            
            // Aquí se implementaría la lógica de filtrado AJAX
            // Por ahora, recargamos la página con parámetros
            const params = new URLSearchParams();
            if (type !== 'all') params.append('type', type);
            if (action !== 'all') params.append('action', action);
            if (date) params.append('date', date);
            if (search) params.append('search', search);
            
            window.location.href = window.location.pathname + '?' + params.toString();
        }
        
        function exportLogs() {
            // Implementar exportación de logs
            alert('Función de exportación en desarrollo');
        }
        
        // Búsqueda en tiempo real
        document.getElementById('searchLogs').addEventListener('input', function() {
            const search = this.value.toLowerCase();
            const rows = document.querySelectorAll('#logsTable tbody tr');
            
            rows.forEach(row => {
                const text = row.textContent.toLowerCase();
                row.style.display = text.includes(search) ? '' : 'none';
            });
        });
        </script>
        <?php
    }
}
?>
