<?php
/**
 * Utilidades para geolocalización usando ip-api.com
 * Rate limit: 45 requests per minute
 */

class GeolocationUtils {
    private $cache = [];
    private $cacheFile;
    private $lastRequestTime = 0;
    private $requestCount = 0;
    private $rateLimitWindow = 60; // 60 segundos
    private $maxRequests = 45; // 45 requests por minuto
    
    public function __construct() {
        $this->cacheFile = sys_get_temp_dir() . '/geolocation_cache.json';
        $this->loadCache();
    }
    
    /**
     * Obtiene información completa de geolocalización de una IP usando la API de ip-api.com
     */
    public function getGeolocationFromIP($ip) {
        // Validar IP
        if (!$this->isValidIP($ip)) {
            return ['error' => 'IP inválida'];
        }
        
        // Verificar cache
        if (isset($this->cache[$ip])) {
            return $this->cache[$ip];
        }
        
        // Respeta rate limiting
        $this->respectRateLimit();
        
        // Consultar API con todos los campos disponibles
        $fields = 'status,message,country,countryCode,region,regionName,city,zip,lat,lon,timezone,isp,org,as,query';
        $url = "http://ip-api.com/json/{$ip}?fields={$fields}";
        $response = $this->makeRequest($url);
        
        if ($response === false) {
            return ['error' => 'Error al consultar la API'];
        }
        
        $data = json_decode($response, true);
        
        if (!$data || $data['status'] !== 'success') {
            $result = [
                'error' => $data['message'] ?? 'Error desconocido',
                'country' => 'Desconocido',
                'countryCode' => 'XX',
                'region' => '',
                'regionName' => '',
                'city' => '',
                'zip' => '',
                'lat' => null,
                'lon' => null,
                'timezone' => '',
                'isp' => '',
                'org' => '',
                'as' => '',
                'query' => $ip
            ];
        } else {
            $result = [
                'country' => $data['country'] ?? 'Desconocido',
                'countryCode' => $data['countryCode'] ?? 'XX',
                'region' => $data['region'] ?? '',
                'regionName' => $data['regionName'] ?? '',
                'city' => $data['city'] ?? '',
                'zip' => $data['zip'] ?? '',
                'lat' => isset($data['lat']) ? floatval($data['lat']) : null,
                'lon' => isset($data['lon']) ? floatval($data['lon']) : null,
                'timezone' => $data['timezone'] ?? '',
                'isp' => $data['isp'] ?? '',
                'org' => $data['org'] ?? '',
                'as' => $data['as'] ?? '',
                'query' => $data['query'] ?? $ip,
                'error' => null
            ];
        }
        
        // Guardar en cache
        $this->cache[$ip] = $result;
        $this->saveCache();
        
        return $result;
    }
    
    /**
     * Obtiene el país de una IP usando la API de ip-api.com (método legacy)
     */
    public function getCountryFromIP($ip) {
        $data = $this->getGeolocationFromIP($ip);
        
        // Retornar solo la información básica para compatibilidad
        return [
            'country' => $data['country'] ?? 'Desconocido',
            'countryCode' => $data['countryCode'] ?? 'XX',
            'error' => $data['error'] ?? null
        ];
    }
    
    /**
     * Obtiene estadísticas de países desde la base de datos
     */
    public function getCountryStats($db_jade, $limit = null, $today_only = false) {
        // Si no se especifica today_only, obtener datos desde la base de datos
        if (!$today_only) {
            return $this->getCountryStatsFromDatabase($db_jade);
        }
        
        // Si se especifica today_only, procesar nuevas IPs y combinar con datos existentes
        return $this->processNewIPsAndGetStats($db_jade, $limit);
    }
    
    /**
     * Obtiene estadísticas de países directamente desde la tabla geolocation_data
     */
    public function getCountryStatsFromDatabase($db_jade) {
        try {
            $sql = "SELECT 
                        country,
                        country_code,
                        COUNT(*) as count
                    FROM geolocation_data 
                    WHERE country IS NOT NULL AND country != ''
                    GROUP BY country, country_code 
                    ORDER BY count DESC";
            
            $stmt = $db_jade->query($sql);
            $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $totalIPs = 0;
            $countries = [];
            
            foreach ($results as $row) {
                $countryCode = $row['country_code'];
                $countries[$countryCode] = [
                    'country' => $row['country'],
                    'countryCode' => $countryCode,
                    'count' => intval($row['count'])
                ];
                $totalIPs += intval($row['count']);
            }
            
            // Calcular porcentajes
            foreach ($countries as &$country) {
                $country['percentage'] = round(($country['count'] / $totalIPs) * 100, 2);
            }
            
            return [
                'countries' => $countries,
                'totalIPs' => $totalIPs,
                'processed' => $totalIPs,
                'source' => 'database'
            ];
            
        } catch (Exception $e) {
            return [
                'error' => 'Error al obtener estadísticas desde la base de datos: ' . $e->getMessage(),
                'countries' => [],
                'totalIPs' => 0,
                'processed' => 0,
                'source' => 'database'
            ];
        }
    }
    
    /**
     * Procesa nuevas IPs y obtiene estadísticas combinadas
     */
    public function processNewIPsAndGetStats($db_jade, $limit = null) {
        // Configurar límites de memoria y tiempo
        ini_set('memory_limit', '512M');
        set_time_limit(600); // 10 minutos máximo
        
        // Configurar buffer de salida para mejor control
        if (ob_get_level() == 0) {
            ob_start();
        }
        
        $stats = [];
        
        // Obtener IPs únicas usando exactamente la misma lógica que "Actividad Hoy"
        // Modo "Solo Hoy": Procesar solo las IPs del día que NO están en geolocation_data
        $query = "SELECT DISTINCT a.lastip 
                  FROM accounts a
                  WHERE DATE(a.last_active) = CURRENT_DATE
                  AND a.lastip NOT IN (
                      SELECT gd.ip 
                      FROM geolocation_data gd
                  )";
        
        try {
            $stmt = $db_jade->query($query);
            $ips = $stmt->fetchAll(PDO::FETCH_COLUMN);
            
            // Obtener estadísticas de optimización
            // Obtener total de IPs únicas de hoy (sin filtrar)
            $totalQuery = "SELECT COUNT(DISTINCT lastip) as total 
                          FROM accounts 
                          WHERE DATE(last_active) = CURRENT_DATE";
            $totalStmt = $db_jade->query($totalQuery);
            $totalIPsToday = $totalStmt->fetchColumn();
            
            // Obtener IPs ya procesadas hoy
            $processedQuery = "SELECT COUNT(DISTINCT gd.ip) as processed 
                              FROM geolocation_data gd
                              INNER JOIN accounts a ON gd.ip = a.lastip
                              WHERE DATE(a.last_active) = CURRENT_DATE";
            $processedStmt = $db_jade->query($processedQuery);
            $processedIPsToday = $processedStmt->fetchColumn();
            
            // Calcular IPs evitadas
            $avoidedIPs = $totalIPsToday - count($ips);
            
            // Si no hay IPs nuevas para procesar, obtener datos desde la base de datos
            if (count($ips) == 0) {
                echo "<div style='background: rgba(76, 175, 80, 0.2); color: #ffffff; padding: 15px; border-radius: 8px; margin: 10px 0; border: 1px solid rgba(76, 175, 80, 0.3);'>";
                echo "🎉 <strong>¡No hay IPs nuevas para procesar!</strong><br>";
                echo "Todas las IPs de hoy ya están en la base de datos.<br>";
                echo "Obteniendo estadísticas desde la base de datos...";
                echo "</div>";
                
                return $this->getCountryStatsFromDatabase($db_jade);
            }
        } catch (Exception $e) {
            return [
                'error' => 'Error al obtener IPs: ' . $e->getMessage(),
                'countries' => [],
                'totalIPs' => 0,
                'processed' => 0
            ];
        }
        
        // Inicializar o recuperar datos acumulados de sesión
        if (isset($_SESSION['geolocation_partial_data'])) {
            $partialData = $_SESSION['geolocation_partial_data'];
            $storedQueryHash = $partialData['query_hash'] ?? '';
            $currentQueryHash = md5($query);
            
            // Verificar que la consulta sea la misma para continuar
            if ($storedQueryHash === $currentQueryHash) {
                $countryCounts = $partialData['countries'] ?? [];
                $processed = $partialData['processed'] ?? 0;
                $lastBatchIndex = $partialData['last_batch'] ?? -1;
            } else {
                // Si la consulta cambió, reiniciar
                $countryCounts = [];
                $processed = 0;
                $lastBatchIndex = -1;
            }
        } else {
            $countryCounts = [];
            $processed = 0;
            $lastBatchIndex = -1;
        }
        
        $totalIPs = count($ips);
        
        // Procesar en lotes progresivos de 45 IPs por minuto
        $batchSize = 45; // Máximo 45 requests por minuto
        $totalBatches = ceil($totalIPs / $batchSize);
        
        echo "<div style='background: rgba(255, 255, 255, 0.1); color: #ffffff; padding: 15px; border-radius: 8px; margin: 10px 0; border: 1px solid rgba(255, 255, 255, 0.2);'>";
        echo "<strong>🔄 Procesamiento Progresivo Automático</strong><br>";
        echo "Total IPs encontradas: <strong>{$totalIPs}</strong> | Lotes necesarios: <strong>{$totalBatches}</strong> | Tamaño por lote: <strong>{$batchSize}</strong><br>";
        echo "⏱️ Tiempo estimado: <strong>" . ($totalBatches - 1) . " minutos</strong><br>";
        
        // Mostrar estadísticas de optimización
        echo "📊 <strong>Estadísticas de hoy:</strong> Total IPs únicas: <strong>{$totalIPsToday}</strong> | Ya procesadas: <strong>{$processedIPsToday}</strong><br>";
        echo "📅 <strong>Modo:</strong> Solo Hoy (Optimizado) - Procesando solo IPs NUEVAS de hoy<br>";
        
        if (isset($avoidedIPs) && $avoidedIPs > 0) {
            echo "🚀 <strong>Optimización:</strong> <span style='color: #4caf50;'>Se evitaron {$avoidedIPs} IPs ya procesadas</span><br>";
            echo "💾 <strong>IPs ya en cache:</strong> {$processedIPsToday} | <strong>IPs nuevas:</strong> {$totalIPs}<br>";
        }
        
        if ($processed > 0) {
            echo "📊 <strong>Progreso acumulado:</strong> {$processed} IPs ya procesadas<br>";
            echo "🔄 <strong>Continuando desde:</strong> Lote " . ($lastBatchIndex + 2) . "<br>";
            echo "💾 <strong>Estado guardado:</strong> Datos recuperados de sesión anterior<br>";
        }
        echo "</div>";
        
        // Forzar salida inicial
        if (ob_get_level()) {
            ob_flush();
        }
        flush();
        
        for ($batchIndex = $lastBatchIndex + 1; $batchIndex < $totalBatches; $batchIndex++) {
            $startIndex = $batchIndex * $batchSize;
            $endIndex = min($startIndex + $batchSize, $totalIPs);
            $currentBatch = array_slice($ips, $startIndex, $batchSize);
            
            echo "<div style='background: rgba(33, 150, 243, 0.2); color: #ffffff; padding: 10px; border-radius: 5px; margin: 5px 0; border-left: 4px solid #2196f3; border: 1px solid rgba(33, 150, 243, 0.3);'>";
            echo "<strong>📦 Lote " . ($batchIndex + 1) . "/{$totalBatches}</strong> - ";
            echo "Procesando IPs " . ($startIndex + 1) . " a {$endIndex} de {$totalIPs}<br>";
            echo "</div>";
            flush();
            
            foreach ($currentBatch as $ipIndex => $ip) {
                // Verificar si hemos excedido el tiempo límite (optimizado para procesar más IPs por sesión)
                $elapsedTime = time() - $_SERVER['REQUEST_TIME'];
                if ($elapsedTime > 210) { // 3.5 minutos máximo (permite ~157 IPs por sesión)
                    echo "<div style='background: rgba(255, 193, 7, 0.2); color: #ffffff; padding: 10px; border-radius: 5px; margin: 5px 0; border-left: 4px solid #ffc107; border: 1px solid rgba(255, 193, 7, 0.3);'>";
                    echo "⚠️ <strong>Tiempo límite alcanzado</strong> - Procesamiento detenido por seguridad<br>";
                    echo "IPs procesadas: <strong>{$processed}</strong> de <strong>{$totalIPs}</strong><br>";
                    echo "Tiempo transcurrido: <strong>{$elapsedTime}</strong> segundos";
                    echo "</div>";
                    
                    // Guardar estado parcial antes de salir
                    $_SESSION['geolocation_partial_data'] = [
                        'countries' => $countryCounts,
                        'processed' => $processed,
                        'last_batch' => $batchIndex,
                        'total_ips' => $totalIPs,
                        'timestamp' => time(),
                        'timeout' => true
                    ];
                    
                    if (ob_get_level()) {
                        ob_flush();
                    }
                    flush();
                    break 2;
                }
                
                // Obtener datos completos de geolocalización
                $geolocationData = $this->getGeolocationFromIP($ip);
                
                // Guardar en base de datos si no hay error
                if (!isset($geolocationData['error'])) {
                    $this->saveGeolocationData($db_jade, $ip, $geolocationData);
                    
                    $country = $geolocationData['country'];
                    $countryCode = $geolocationData['countryCode'];
                    
                    if (!isset($countryCounts[$countryCode])) {
                        $countryCounts[$countryCode] = [
                            'country' => $country,
                            'countryCode' => $countryCode,
                            'count' => 0
                        ];
                    }
                    $countryCounts[$countryCode]['count']++;
                }
                
                $processed++;
                
                // Mostrar progreso detallado (reducir frecuencia de flush aún más)
                if (($processed % 10 == 0) || ($processed == $totalIPs)) {
                    $percentage = round(($processed / $totalIPs) * 100, 1);
                    echo "<div style='background: rgba(76, 175, 80, 0.2); color: #ffffff; padding: 8px; border-radius: 3px; margin: 2px 0; border-left: 3px solid #4caf50; border: 1px solid rgba(76, 175, 80, 0.3);'>";
                    echo "✅ <strong>{$processed}/{$totalIPs}</strong> IPs procesadas ({$percentage}%) - ";
                    echo "Lote " . ($batchIndex + 1) . "/{$totalBatches} - ";
                    echo "Países acumulados: <strong>" . count($countryCounts) . "</strong>";
                    echo "</div>";
                    
                    // Solo hacer flush cada 20 IPs para reducir sobrecarga
                    if ($processed % 20 == 0) {
                        if (ob_get_level()) {
                            ob_flush();
                        }
                        flush();
                    }
                }
            }
            
            // Guardar datos acumulados en sesión después de cada lote
            $_SESSION['geolocation_partial_data'] = [
                'countries' => $countryCounts,
                'processed' => $processed,
                'last_batch' => $batchIndex,
                'total_ips' => $totalIPs,
                'timestamp' => time(),
                'query_hash' => md5($query) // Para verificar que la consulta no cambió
            ];
            
            // Forzar guardado de sesión inmediatamente (sin session_start() problemático)
            session_write_close();
            
            // Pausa de 1 minuto entre lotes (excepto en el último lote)
            if ($batchIndex < $totalBatches - 1) {
                echo "<div style='background: rgba(255, 152, 0, 0.2); color: #ffffff; padding: 10px; border-radius: 5px; margin: 5px 0; border-left: 4px solid #ff9800; border: 1px solid rgba(255, 152, 0, 0.3);'>";
                echo "⏳ <strong>Esperando 1 minuto</strong> para respetar rate limit de la API...<br>";
                echo "Próximo lote: " . ($batchIndex + 2) . "/{$totalBatches}<br>";
                echo "💾 <strong>Datos guardados:</strong> {$processed} IPs procesadas, " . count($countryCounts) . " países acumulados";
                echo "</div>";
                
                // Forzar salida antes de la pausa
                if (ob_get_level()) {
                    ob_flush();
                }
                flush();
                sleep(60); // 1 minuto
            }
        }
        
        // Limpiar datos parciales de sesión al completar
        unset($_SESSION['geolocation_partial_data']);
        
        echo "<div style='background: rgba(76, 175, 80, 0.2); color: #ffffff; padding: 15px; border-radius: 8px; margin: 10px 0; border-left: 4px solid #4caf50; border: 1px solid rgba(76, 175, 80, 0.3);'>";
        echo "🎉 <strong>¡Procesamiento completado!</strong><br>";
        echo "Total IPs procesadas: <strong>{$processed}</strong> de <strong>{$totalIPs}</strong><br>";
        echo "Países detectados: <strong>" . count($countryCounts) . "</strong><br>";
        echo "📊 <strong>Datos acumulados correctamente</strong> - Todos los lotes procesados";
        echo "</div>";
        
        // Forzar la salida y terminar el procesamiento
        if (ob_get_level()) {
            ob_end_flush();
        }
        flush();
        
        // Verificar que los datos se guarden correctamente en sesión
        if (session_status() === PHP_SESSION_ACTIVE) {
            session_write_close();
        }
        
        // Obtener datos existentes de la base de datos
        $existingStats = $this->getCountryStatsFromDatabase($db_jade);
        $existingCountries = $existingStats['countries'] ?? [];
        
        // Combinar datos procesados con datos existentes
        foreach ($countryCounts as $countryCode => $newData) {
            if (isset($existingCountries[$countryCode])) {
                $existingCountries[$countryCode]['count'] += $newData['count'];
            } else {
                $existingCountries[$countryCode] = $newData;
            }
        }
        
        // Recalcular totales
        $totalIPs = $existingStats['totalIPs'] + $processed;
        
        // Recalcular porcentajes
        foreach ($existingCountries as &$country) {
            $country['percentage'] = round(($country['count'] / $totalIPs) * 100, 2);
        }
        
        // Ordenar por cantidad de jugadores (descendente)
        uasort($existingCountries, function($a, $b) {
            return $b['count'] - $a['count'];
        });
        
        // Asegurar que el buffer se limpie al final
        if (ob_get_level() > 0) {
            ob_end_flush();
        }
        
        return [
            'countries' => $existingCountries,
            'totalIPs' => $totalIPs,
            'processed' => $processed,
            'source' => 'combined',
            'newIPsProcessed' => $processed,
            'existingIPs' => $existingStats['totalIPs']
        ];
    }
    
    /**
     * Valida si una IP es válida
     */
    private function isValidIP($ip) {
        return filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE);
    }
    
    /**
     * Respeta el rate limiting de la API
     */
    private function respectRateLimit() {
        $currentTime = time();
        
        // Si estamos en una nueva ventana de tiempo, resetear contador
        if ($currentTime - $this->lastRequestTime >= $this->rateLimitWindow) {
            $this->requestCount = 0;
            $this->lastRequestTime = $currentTime;
        }
        
        // Si hemos alcanzado el límite, esperar
        if ($this->requestCount >= $this->maxRequests) {
            $waitTime = $this->rateLimitWindow - ($currentTime - $this->lastRequestTime);
            if ($waitTime > 0) {
                sleep($waitTime);
                $this->requestCount = 0;
                $this->lastRequestTime = time();
            }
        }
        
        $this->requestCount++;
    }
    
    /**
     * Realiza una petición HTTP
     */
    private function makeRequest($url) {
        $context = stream_context_create([
            'http' => [
                'timeout' => 10,
                'user_agent' => 'Lineage2-Panel/1.0'
            ]
        ]);
        
        $response = @file_get_contents($url, false, $context);
        
        if ($response === false) {
            return false;
        }
        
        return $response;
    }
    
    /**
     * Carga el cache desde archivo
     */
    private function loadCache() {
        if (file_exists($this->cacheFile)) {
            $cacheData = file_get_contents($this->cacheFile);
            $this->cache = json_decode($cacheData, true) ?: [];
        }
    }
    
    /**
     * Guarda el cache en archivo
     */
    private function saveCache() {
        file_put_contents($this->cacheFile, json_encode($this->cache));
    }
    
    /**
     * Limpia el cache
     */
    public function clearCache() {
        $this->cache = [];
        if (file_exists($this->cacheFile)) {
            unlink($this->cacheFile);
        }
    }
    
    /**
     * Guarda datos de geolocalización en la base de datos
     */
    public function saveGeolocationData($db, $ip, $data) {
        try {
            $sql = "INSERT INTO geolocation_data 
                    (ip, country, country_code, region, region_name, city, zip, lat, lon, timezone, isp, org, as_info, query) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                    ON DUPLICATE KEY UPDATE 
                    country = VALUES(country),
                    country_code = VALUES(country_code),
                    region = VALUES(region),
                    region_name = VALUES(region_name),
                    city = VALUES(city),
                    zip = VALUES(zip),
                    lat = VALUES(lat),
                    lon = VALUES(lon),
                    timezone = VALUES(timezone),
                    isp = VALUES(isp),
                    org = VALUES(org),
                    as_info = VALUES(as_info),
                    updated_at = CURRENT_TIMESTAMP";
            
            $stmt = $db->prepare($sql);
            $stmt->execute([
                $ip,
                $data['country'],
                $data['countryCode'],
                $data['region'],
                $data['regionName'],
                $data['city'],
                $data['zip'],
                $data['lat'],
                $data['lon'],
                $data['timezone'],
                $data['isp'],
                $data['org'],
                $data['as'],
                $data['query']
            ]);
            
            return true;
        } catch (Exception $e) {
            error_log("Error saving geolocation data: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Obtiene datos de geolocalización desde la base de datos
     */
    public function getGeolocationFromDatabase($db, $ip) {
        try {
            $sql = "SELECT * FROM geolocation_data WHERE ip = ?";
            $stmt = $db->prepare($sql);
            $stmt->execute([$ip]);
            
            $data = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($data) {
                return [
                    'country' => $data['country'],
                    'countryCode' => $data['country_code'],
                    'region' => $data['region'],
                    'regionName' => $data['region_name'],
                    'city' => $data['city'],
                    'zip' => $data['zip'],
                    'lat' => $data['lat'],
                    'lon' => $data['lon'],
                    'timezone' => $data['timezone'],
                    'isp' => $data['isp'],
                    'org' => $data['org'],
                    'as' => $data['as'],
                    'query' => $data['query'],
                    'error' => null
                ];
            }
            
            return null;
        } catch (Exception $e) {
            error_log("Error getting geolocation from database: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Obtiene estadísticas detalladas de geolocalización
     */
    public function getDetailedStats($db) {
        try {
            $stats = [];
            
            // Estadísticas por país
            $sql = "SELECT country, country_code, COUNT(*) as count 
                    FROM geolocation_data 
                    GROUP BY country, country_code 
                    ORDER BY count DESC";
            $stmt = $db->query($sql);
            $stats['countries'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Estadísticas por región
            $sql = "SELECT region_name, COUNT(*) as count 
                    FROM geolocation_data 
                    WHERE region_name != '' 
                    GROUP BY region_name 
                    ORDER BY count DESC 
                    LIMIT 20";
            $stmt = $db->query($sql);
            $stats['regions'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Estadísticas por ciudad (incluyendo todas las ciudades brasileñas importantes)
            $sql = "SELECT city, COUNT(*) as count 
                    FROM geolocation_data 
                    WHERE city != '' 
                    GROUP BY city 
                    ORDER BY count DESC 
                    LIMIT 50"; // Aumentado a 50 para incluir más ciudades
            $stmt = $db->query($sql);
            $stats['cities'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Estadísticas específicas de ciudades brasileñas
            $sql = "SELECT city, COUNT(*) as count 
                    FROM geolocation_data 
                    WHERE country_code = 'BR' AND city != '' 
                    GROUP BY city 
                    ORDER BY count DESC 
                    LIMIT 20";
            $stmt = $db->query($sql);
            $stats['brazil_cities'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Estadísticas específicas de ciudades argentinas
            $sql = "SELECT city, COUNT(*) as count 
                    FROM geolocation_data 
                    WHERE country_code = 'AR' AND city != '' 
                    GROUP BY city 
                    ORDER BY count DESC 
                    LIMIT 20";
            $stmt = $db->query($sql);
            $stats['argentina_cities'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Estadísticas por ISP
            $sql = "SELECT isp, COUNT(*) as count 
                    FROM geolocation_data 
                    WHERE isp != '' 
                    GROUP BY isp 
                    ORDER BY count DESC 
                    LIMIT 20";
            $stmt = $db->query($sql);
            $stats['isps'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Estadísticas por zona horaria
            $sql = "SELECT timezone, COUNT(*) as count 
                    FROM geolocation_data 
                    WHERE timezone != '' 
                    GROUP BY timezone 
                    ORDER BY count DESC 
                    LIMIT 10";
            $stmt = $db->query($sql);
            $stats['timezones'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Total de registros
            $sql = "SELECT COUNT(*) as total FROM geolocation_data";
            $stmt = $db->query($sql);
            $stats['total'] = $stmt->fetchColumn();
            
            return $stats;
        } catch (Exception $e) {
            error_log("Error getting detailed stats: " . $e->getMessage());
            return ['error' => $e->getMessage()];
        }
    }
    
    /**
     * Obtiene coordenadas para el mapa
     */
    public function getMapCoordinates($db, $limit = 100) {
        try {
            $sql = "SELECT lat, lon, city, country, COUNT(*) as count 
                    FROM geolocation_data 
                    WHERE lat IS NOT NULL AND lon IS NOT NULL 
                    GROUP BY lat, lon, city, country 
                    ORDER BY count DESC 
                    LIMIT ?";
            $stmt = $db->prepare($sql);
            $stmt->execute([$limit]);
            
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            error_log("Error getting map coordinates: " . $e->getMessage());
            return [];
        }
    }
}
?>
