<?php
session_start();

// Verificar si el usuario está logueado
if (!isset($_SESSION['user_id'])) {
    header("Location: index.php");
    exit();
}

require_once "config/database_advanced.php";
require_once "includes/log_viewer.php";

// Inicializar el gestor de base de datos avanzado
$dbManager = AdvancedDatabaseManager::getInstance();

// Obtener estadísticas actuales
$stats = $dbManager->getAdvancedStats();
$health = $dbManager->checkConnectionHealth();

// Inicializar el visualizador de logs
$logViewer = new LogViewer();
$logs = $logViewer->loadLogs();

// Aplicar filtros si se proporcionan
$filters = [];
if (!empty($_GET['type'])) $filters['type'] = $_GET['type'];
if (!empty($_GET['action'])) $filters['action'] = $_GET['action'];
if (!empty($_GET['date'])) $filters['date'] = $_GET['date'];
if (!empty($_GET['search'])) $filters['search'] = $_GET['search'];

$filteredLogs = $logViewer->filterLogs($filters);
$logStats = $logViewer->getLogStats();

// Obtener información del sistema
$systemInfo = [
    'php_version' => PHP_VERSION,
    'memory_limit' => ini_get('memory_limit'),
    'max_execution_time' => ini_get('max_execution_time'),
    'server_time' => date('Y-m-d H:i:s'),
    'uptime' => function_exists('sys_getloadavg') ? sys_getloadavg() : 'N/A'
];
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Herramientas de Monitoreo - Panel de Estadísticas</title>
    <link rel="stylesheet" href="assets/css/dark-theme.css">
    <link rel="stylesheet" href="assets/css/monitoring.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        /* Estilos dark específicos para herramientas de monitoreo */
        .monitoring-container {
            background: linear-gradient(135deg, #1a1a1a 0%, #2d2d2d 100%);
            min-height: 100vh;
            padding: 20px;
        }

        .monitoring-header {
            background: rgba(45, 45, 45, 0.95);
            border-radius: 15px;
            padding: 25px;
            margin-bottom: 25px;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.3);
            backdrop-filter: blur(10px);
            border: 1px solid #404040;
        }

        .monitoring-title {
            color: #ffffff;
            font-size: 2.5rem;
            font-weight: 700;
            margin-bottom: 10px;
            text-align: center;
        }

        .monitoring-subtitle {
            color: #b0b0b0;
            text-align: center;
            font-size: 1.1rem;
        }

        .monitoring-card {
            background: rgba(45, 45, 45, 0.95);
            border-radius: 15px;
            padding: 25px;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.3);
            backdrop-filter: blur(10px);
            transition: transform 0.3s ease, box-shadow 0.3s ease;
            border: 1px solid #404040;
            color: #e0e0e0;
        }

        .monitoring-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 40px rgba(0, 0, 0, 0.4);
            border-color: #505050;
        }

        .card-header {
            display: flex;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid #404040;
        }

        .card-title {
            font-size: 1.4rem;
            font-weight: 600;
            color: #ffffff;
            margin: 0;
        }

        .status-indicator {
            display: inline-flex;
            align-items: center;
            padding: 8px 15px;
            border-radius: 25px;
            font-weight: 600;
            font-size: 0.9rem;
            margin: 5px;
        }

        .status-active {
            background: linear-gradient(45deg, #00b894, #00cec9);
            color: white;
        }

        .status-inactive {
            background: linear-gradient(45deg, #e17055, #d63031);
            color: white;
        }

        .status-warning {
            background: linear-gradient(45deg, #fdcb6e, #e17055);
            color: white;
        }

        .btn-monitoring {
            padding: 10px 20px;
            border: none;
            border-radius: 8px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            font-size: 0.9rem;
        }

        .btn-primary {
            background: linear-gradient(45deg, #0d6efd, #0b5ed7);
            color: white;
        }

        .btn-success {
            background: linear-gradient(45deg, #00b894, #00cec9);
            color: white;
        }

        .btn-warning {
            background: linear-gradient(45deg, #fdcb6e, #e17055);
            color: white;
        }

        .btn-danger {
            background: linear-gradient(45deg, #e17055, #d63031);
            color: white;
        }

        .btn-monitoring:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.4);
        }

        .logs-container {
            background: rgba(45, 45, 45, 0.95);
            border-radius: 15px;
            padding: 25px;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.3);
            backdrop-filter: blur(10px);
            margin-bottom: 25px;
            border: 1px solid #404040;
        }

        .logs-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
            background: #2d2d2d;
            border-radius: 10px;
            overflow: hidden;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.3);
        }

        .logs-table th {
            background: linear-gradient(45deg, #0d6efd, #0b5ed7);
            color: white;
            padding: 15px;
            text-align: left;
            font-weight: 600;
            font-size: 0.95rem;
        }

        .logs-table td {
            padding: 12px 15px;
            border-bottom: 1px solid #404040;
            font-size: 0.9rem;
            color: #e0e0e0;
        }

        .logs-table tr:hover {
            background: #353535;
        }

        .logs-table tr:last-child td {
            border-bottom: none;
        }

        .filter-group {
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .filter-label {
            font-weight: 600;
            color: #ffffff;
            font-size: 0.9rem;
        }

        .filter-select, .filter-input {
            padding: 8px 12px;
            border: 2px solid #404040;
            border-radius: 8px;
            font-size: 0.9rem;
            transition: border-color 0.3s ease;
            background: #2d2d2d;
            color: #e0e0e0;
        }

        .filter-select:focus, .filter-input:focus {
            outline: none;
            border-color: #0d6efd;
            background: #353535;
        }

        .logs-controls {
            display: flex;
            gap: 15px;
            margin-bottom: 20px;
            flex-wrap: wrap;
            align-items: center;
        }

        .search-box {
            flex: 1;
            min-width: 200px;
        }

        /* Scrollbar personalizado para dark theme */
        .logs-container::-webkit-scrollbar {
            width: 8px;
        }

        .logs-container::-webkit-scrollbar-track {
            background: #2d2d2d;
            border-radius: 4px;
        }

        .logs-container::-webkit-scrollbar-thumb {
            background: #0d6efd;
            border-radius: 4px;
        }

        .logs-container::-webkit-scrollbar-thumb:hover {
            background: #0b5ed7;
        }

        /* Responsive */
        @media (max-width: 768px) {
            .monitoring-container {
                padding: 15px;
            }
            
            .monitoring-title {
                font-size: 2rem;
            }
            
            .monitoring-grid {
                grid-template-columns: 1fr;
            }
            
            .logs-controls {
                flex-direction: column;
                align-items: stretch;
            }
            
            .search-box {
                min-width: auto;
            }
            
            .logs-table {
                font-size: 0.8rem;
            }
            
            .logs-table th,
            .logs-table td {
                padding: 8px 10px;
            }
        }
    </style>
</head>
<body>
    <div class="monitoring-container">
        <!-- Header -->
        <div class="monitoring-header">
            <h1 class="monitoring-title">
                <i class="fas fa-tools"></i> Herramientas de Monitoreo
            </h1>
            <p class="monitoring-subtitle">
                Panel de control para monitorear y gestionar las conexiones de base de datos
            </p>
            <div style="text-align: center; margin-top: 15px;">
                <a href="dashboard.php" class="btn-monitoring btn-primary">
                    <i class="fas fa-arrow-left"></i> Volver al Dashboard
                </a>
            </div>
        </div>

        <!-- Estado del Sistema -->
        <div class="monitoring-grid">
            <div class="monitoring-card">
                <div class="card-header">
                    <div class="card-icon" style="background: linear-gradient(45deg, #667eea, #764ba2);">
                        <i class="fas fa-database"></i>
                    </div>
                    <h3 class="card-title">Estado de Conexiones</h3>
                </div>
                
                <div style="margin-bottom: 20px;">
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span style="color: #e0e0e0;"><strong>Conexión CRM:</strong></span>
                        <span class="status-indicator <?php echo $stats['crm_connected'] ? 'status-active' : 'status-inactive'; ?>">
                            <?php echo $stats['crm_connected'] ? '✅ Activa' : '❌ Inactiva'; ?>
                        </span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span style="color: #e0e0e0;"><strong>Conexión Jade:</strong></span>
                        <span class="status-indicator <?php echo $stats['jade_connected'] ? 'status-active' : 'status-inactive'; ?>">
                            <?php echo $stats['jade_connected'] ? '✅ Activa' : '❌ Inactiva'; ?>
                        </span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span style="color: #e0e0e0;"><strong>Total Conexiones:</strong></span>
                        <span class="status-indicator status-active">
                            <?php echo $stats['total_connections']; ?>/<?php echo $stats['max_connections']; ?>
                        </span>
                    </div>
                </div>
                
                <div style="text-align: center;">
                    <a href="verificar_conexiones_abiertas.php" class="btn-monitoring btn-primary" target="_blank">
                        <i class="fas fa-search"></i> Verificar Conexiones
                    </a>
                </div>
            </div>

            <div class="monitoring-card">
                <div class="card-header">
                    <div class="card-icon" style="background: linear-gradient(45deg, #00b894, #00cec9);">
                        <i class="fas fa-heartbeat"></i>
                    </div>
                    <h3 class="card-title">Health Check</h3>
                </div>
                
                <div style="margin-bottom: 20px;">
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span style="color: #e0e0e0;"><strong>CRM:</strong></span>
                        <span class="status-indicator <?php echo $health['crm']['status'] === 'healthy' ? 'status-active' : 'status-warning'; ?>">
                            <?php echo strtoupper($health['crm']['status']); ?>
                        </span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span style="color: #e0e0e0;"><strong>Jade:</strong></span>
                        <span class="status-indicator <?php echo $health['jade']['status'] === 'healthy' ? 'status-active' : 'status-warning'; ?>">
                            <?php echo strtoupper($health['jade']['status']); ?>
                        </span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span style="color: #e0e0e0;"><strong>Timeout:</strong></span>
                        <span style="color: #e0e0e0;"><?php echo $stats['connection_timeout']; ?>s</span>
                    </div>
                </div>
                
                <div style="text-align: center;">
                    <a href="verificar_conexiones_sistema.php" class="btn-monitoring btn-success" target="_blank">
                        <i class="fas fa-server"></i> Verificar Sistema
                    </a>
                </div>
            </div>

            <div class="monitoring-card">
                <div class="card-header">
                    <div class="card-icon" style="background: linear-gradient(45deg, #fdcb6e, #e17055);">
                        <i class="fas fa-chart-line"></i>
                    </div>
                    <h3 class="card-title">Métricas de Rendimiento</h3>
                </div>
                
                <div style="margin-bottom: 20px;">
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span style="color: #e0e0e0;"><strong>Tiempo Promedio:</strong></span>
                        <span style="color: #e0e0e0;"><?php 
                            $avgTime = 0;
                            if (isset($stats['performance_summary']['CRM']['connection_time']['avg'])) {
                                $avgTime = $stats['performance_summary']['CRM']['connection_time']['avg'] * 1000;
                            }
                            echo number_format($avgTime, 2); 
                        ?>ms</span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span style="color: #e0e0e0;"><strong>Conexiones Reutilizadas:</strong></span>
                        <span style="color: #e0e0e0;"><?php echo count($stats['recent_logs']); ?></span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span style="color: #e0e0e0;"><strong>Memoria Usada:</strong></span>
                        <span style="color: #e0e0e0;"><?php echo number_format(memory_get_usage(true) / 1024 / 1024, 2); ?> MB</span>
                    </div>
                </div>
                
                <div style="text-align: center;">
                    <a href="monitor_avanzado.php" class="btn-monitoring btn-warning" target="_blank">
                        <i class="fas fa-chart-bar"></i> Ver Métricas
                    </a>
                </div>
            </div>

            <div class="monitoring-card">
                <div class="card-header">
                    <div class="card-icon" style="background: linear-gradient(45deg, #e17055, #d63031);">
                        <i class="fas fa-cogs"></i>
                    </div>
                    <h3 class="card-title">Acciones Rápidas</h3>
                </div>
                
                <div style="margin-bottom: 20px;">
                    <p style="color: #b0b0b0; font-size: 0.9rem; margin-bottom: 15px;">
                        Acciones para gestionar las conexiones de base de datos
                    </p>
                </div>
                
                <div style="display: flex; flex-direction: column; gap: 10px;">
                    <a href="diagnostico_conexiones_detallado.php" class="btn-monitoring btn-warning">
                        <i class="fas fa-search"></i> Diagnóstico Detallado
                    </a>
                    
                    <a href="limpieza_emergencia.php" class="btn-monitoring btn-danger">
                        <i class="fas fa-fire-extinguisher"></i> Limpieza de Emergencia
                    </a>
                    
                    <a href="correccion_conexiones.php" class="btn-monitoring btn-primary">
                        <i class="fas fa-wrench"></i> Corregir Sistema
                    </a>
                    
                    <button class="btn-monitoring btn-success" onclick="refreshStats()">
                        <i class="fas fa-sync-alt"></i> Actualizar Estadísticas
                    </button>
                </div>
            </div>
        </div>

        <!-- Información del Sistema -->
        <div class="monitoring-card">
            <div class="card-header">
                <div class="card-icon" style="background: linear-gradient(45deg, #6c5ce7, #a29bfe);">
                    <i class="fas fa-info-circle"></i>
                </div>
                <h3 class="card-title">Información del Sistema</h3>
            </div>
            
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px;">
                <div style="color: #e0e0e0;">
                    <strong>PHP Version:</strong> <?php echo $systemInfo['php_version']; ?>
                </div>
                <div style="color: #e0e0e0;">
                    <strong>Memory Limit:</strong> <?php echo $systemInfo['memory_limit']; ?>
                </div>
                <div style="color: #e0e0e0;">
                    <strong>Max Execution Time:</strong> <?php echo $systemInfo['max_execution_time']; ?>s
                </div>
                <div style="color: #e0e0e0;">
                    <strong>Server Time:</strong> <?php echo $systemInfo['server_time']; ?>
                </div>
                <?php if ($systemInfo['uptime'] !== 'N/A'): ?>
                <div style="color: #e0e0e0;">
                    <strong>System Load:</strong> <?php echo implode(', ', array_map('number_format', $systemInfo['uptime'])); ?>
                </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Visualizador de Logs -->
        <?php $logViewer->renderLogsTable($filteredLogs); ?>
    </div>

    <script>
    function refreshStats() {
        location.reload();
    }
    
    function runHealthCheck() {
        // Simular health check
        alert('Health check ejecutado. Revisa la consola para más detalles.');
        
        // Aquí se podría implementar una llamada AJAX para ejecutar el health check
        fetch('verificar_conexiones_abiertas.php')
            .then(response => response.text())
            .then(data => {
                console.log('Health check completado');
                location.reload();
            })
            .catch(error => {
                console.error('Error en health check:', error);
            });
    }
    
    // Auto-refresh cada 30 segundos
    setInterval(function() {
        // Solo actualizar si no hay filtros activos
        if (window.location.search === '') {
            refreshStats();
        }
    }, 30000);
    </script>
</body>
</html>
