<?php
session_start();
if(!isset($_SESSION['user_id'])) {
    header("Location: index.php");
    exit();
}

require_once "config/database_singleton.php";
require_once "includes/geolocation_utils.php";

$db = getDB('crm');
$db_jade = getDB('jade');

$geolocation = new GeolocationUtils();

// Manejar acciones
$action = $_GET['action'] ?? '';
$message = '';

if ($action === 'clear_cache') {
    $geolocation->clearCache();
    $message = 'Cache limpiado correctamente';
}

if ($action === 'clear_partial_data') {
    unset($_SESSION['geolocation_partial_data']);
    $message = 'Datos parciales limpiados - Puedes reiniciar el procesamiento';
}

// API endpoint para obtener coordenadas del mapa
if ($action === 'get_coordinates') {
    header('Content-Type: application/json');
    $limit = isset($_GET['limit']) ? intval($_GET['limit']) : 100;
    
    try {
        $coordinates = $geolocation->getMapCoordinates($db_jade, $limit);
        echo json_encode([
            'success' => true,
            'coordinates' => $coordinates
        ]);
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'error' => $e->getMessage()
        ]);
    }
    exit();
}

// API endpoint para obtener estadísticas detalladas
if ($action === 'get_detailed_stats') {
    header('Content-Type: application/json');
    
    try {
        $stats = $geolocation->getDetailedStats($db_jade);
        echo json_encode([
            'success' => true,
            'stats' => $stats
        ]);
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'error' => $e->getMessage()
        ]);
    }
    exit();
}

// Obtener estadísticas de países
$countryStats = null;
$isProcessing = false;

if (isset($_GET['refresh']) || !isset($_SESSION['geolocation_stats'])) {
    $isProcessing = true;
    
    // Obtener estadísticas
    $today_only = isset($_GET['today']); // Solo IPs de hoy si se especifica
    
    // Verificar si hay datos parciales para continuar automáticamente
    $continue_partial = false;
    if (isset($_SESSION['geolocation_partial_data']) && !isset($_GET['continue'])) {
        $partial = $_SESSION['geolocation_partial_data'];
        $timeSinceLastUpdate = time() - ($partial['timestamp'] ?? 0);
        $hasTimeout = isset($_SESSION['geolocation_partial_data']['timeout']);
        
        // Si hay timeout o han pasado más de 10 segundos, continuar automáticamente
        if (($hasTimeout && $timeSinceLastUpdate > 5) || 
            ($timeSinceLastUpdate > 10 && $partial['processed'] < $partial['total_ips'])) {
            $continue_partial = true;
        }
    }
    
            try {
            // Si hay datos parciales y debemos continuar, usar los mismos parámetros
            if ($continue_partial) {
                $partial = $_SESSION['geolocation_partial_data'];
                $hasTimeout = isset($_SESSION['geolocation_partial_data']['timeout']);
                
                // Agregar mensaje de debugging
                if ($hasTimeout) {
                    $message = '🔄 Auto-continuando desde timeout...';
                } else {
                    $message = '🔄 Auto-continuando procesamiento...';
                }
                
                // Determinar el modo basado en los datos guardados
                if (isset($partial['query_hash'])) {
                    // Recrear los parámetros originales
                    $today_only = (strpos($partial['query_hash'], 'CURRENT_DATE') !== false && strpos($partial['query_hash'], 'LIMIT') === false);
                }
            }
            
            $countryStats = $geolocation->getCountryStats($db_jade, null, $today_only);
        
        // Verificar si hay error en la respuesta
        if (isset($countryStats['error'])) {
            $message = 'Error: ' . $countryStats['error'];
            $countryStats = null;
        } else {
            // Guardar en sesión para evitar reprocesamiento
            $_SESSION['geolocation_stats'] = $countryStats;
            $_SESSION['geolocation_timestamp'] = time();
        }
    } catch (Exception $e) {
        $message = 'Error al procesar datos: ' . $e->getMessage();
        $countryStats = null;
    }
    
    $isProcessing = false;
} else {
    $countryStats = $_SESSION['geolocation_stats'] ?? null;
}

// Calcular tiempo desde última actualización
$lastUpdate = '';
if (isset($_SESSION['geolocation_timestamp'])) {
    $timeDiff = time() - $_SESSION['geolocation_timestamp'];
    if ($timeDiff < 60) {
        $lastUpdate = "Hace " . $timeDiff . " segundos";
    } elseif ($timeDiff < 3600) {
        $lastUpdate = "Hace " . round($timeDiff / 60) . " minutos";
    } else {
        $lastUpdate = "Hace " . round($timeDiff / 3600) . " horas";
    }
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>CRM - Geolocalización</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" rel="stylesheet">
    <link href="https://unpkg.com/leaflet.markercluster@1.4.1/dist/MarkerCluster.css" rel="stylesheet">
    <link href="https://unpkg.com/leaflet.markercluster@1.4.1/dist/MarkerCluster.Default.css" rel="stylesheet">
    <link href="assets/css/dark-theme.css" rel="stylesheet">
    <style>
        .country-flag {
            width: 24px;
            height: 16px;
            margin-right: 8px;
            border-radius: 2px;
        }
        .progress-bar-custom {
            height: 8px;
            border-radius: 4px;
        }
        .stats-card {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 20px;
        }
        .refresh-btn {
            background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
            border: none;
            color: white;
            padding: 10px 20px;
            border-radius: 8px;
            transition: all 0.3s ease;
        }
        .refresh-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.3);
        }
        
        #geolocation-map {
            height: 500px;
            width: 100%;
            border-radius: 8px;
            border: 1px solid rgba(255, 255, 255, 0.2);
        }
        
        .map-container {
            background: rgba(255, 255, 255, 0.05);
            border-radius: 12px;
            padding: 20px;
            margin: 20px 0;
        }
        
        .stats-tabs {
            margin-bottom: 20px;
        }
        
        .stats-tabs .nav-link {
            color: #fff;
            border: 1px solid rgba(255, 255, 255, 0.2);
            margin-right: 5px;
            border-radius: 8px 8px 0 0;
        }
        
        .stats-tabs .nav-link.active {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border-color: transparent;
        }
        
        .tab-content {
            background: rgba(255, 255, 255, 0.05);
            border: 1px solid rgba(255, 255, 255, 0.2);
            border-radius: 0 8px 8px 8px;
            padding: 20px;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-2 sidebar">
                <h3 class="text-white text-center mb-4">CRM Panel</h3>
                <a href="dashboard.php"><i class="fas fa-home"></i> Dashboard</a>
                <a href="accounts.php"><i class="fas fa-user-circle"></i> Accounts</a>
                <a href="characters.php"><i class="fas fa-gamepad"></i> Personajes</a>
                <a href="geolocalizacion.php" class="active"><i class="fas fa-globe"></i> Geolocalización</a>
                <?php if (isset($_SESSION['username']) && $_SESSION['username'] === 'jade'): ?>
                <a href="donations.php"><i class="fas fa-hand-holding-usd"></i> Donaciones</a>
                <?php endif; ?>
                <a href="logout.php"><i class="fas fa-sign-out-alt"></i> Cerrar Sesión</a>
            </div>

            <!-- Main Content -->
            <div class="col-md-10 main-content">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2>
                        <i class="fas fa-globe"></i> 
                        Geolocalización de Usuarios
                    </h2>
                    <div>
                        <a href="?refresh=1&today=1" class="btn btn-info me-2">
                            <i class="fas fa-calendar-day"></i> Procesar Nuevas IPs
                        </a>
                    </div>
                </div>

                <?php if (isset($_SESSION['geolocation_partial_data']) && isset($_SESSION['geolocation_partial_data']['timeout'])): ?>
                <div class="alert alert-warning mt-3">
                    <i class="fas fa-exclamation-triangle"></i> <strong>Procesamiento interrumpido por timeout:</strong> 
                    <?php 
                    $partial = $_SESSION['geolocation_partial_data'];
                    $percentage = round(($partial['processed'] / $partial['total_ips']) * 100, 1);
                    $timeSinceLastUpdate = time() - ($partial['timestamp'] ?? 0);
                    echo "{$partial['processed']} IPs procesadas de {$partial['total_ips']} totales ({$percentage}%)";
                    ?>
                    <br><small>El procesamiento se detuvo por límite de tiempo. <strong>Usa el botón "Continuar Ahora" para reanudar.</strong></small>
                    <div class="progress mt-2" style="height: 8px;">
                        <div class="progress-bar bg-warning" style="width: <?php echo $percentage; ?>%"></div>
                    </div>
                    <a href="?refresh=1&continue=1" class="btn btn-sm btn-warning ms-2">
                        <i class="fas fa-play"></i> Continuar Ahora
                    </a>
                    <a href="?refresh=1&today=1" class="btn btn-sm btn-info ms-2">
                        <i class="fas fa-redo"></i> Reiniciar Procesamiento
                    </a>
                </div>
                <?php endif; ?>

                <?php if ($message): ?>
                <div class="alert alert-success alert-dismissible fade show" role="alert">
                    <?php echo htmlspecialchars($message); ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>

                <?php if ($isProcessing): ?>
                <div class="alert alert-info">
                    <i class="fas fa-spinner fa-spin"></i> Procesando datos de geolocalización...
                    <div class="progress mt-2">
                        <div class="progress-bar progress-bar-striped progress-bar-animated" 
                             style="width: 100%"></div>
                    </div>
                </div>
                <?php endif; ?>



                <?php if (isset($_GET['today'])): ?>
                <div class="alert alert-info">
                    <i class="fas fa-calendar-day"></i> <strong>Modo Hoy Optimizado:</strong> Procesando solo las IPs NUEVAS de la actividad de hoy (<?php echo date('d/m/Y'); ?>) que no están en cache.
                    <br><small><i class="fas fa-rocket"></i> <strong>Optimización inteligente:</strong> Se evitan automáticamente las IPs que ya fueron procesadas anteriormente.</small>
                    <br><small><i class="fas fa-info-circle"></i> <strong>Procesamiento progresivo:</strong> 45 IPs por minuto automáticamente hasta completar las IPs nuevas de hoy.</small>
                    <br><small><i class="fas fa-clock"></i> <strong>Tiempo estimado:</strong> Depende del total de IPs nuevas de hoy (se mostrará al iniciar).</small>
                    <br><small><i class="fas fa-sync-alt"></i> <strong>Auto-continuación:</strong> Si se interrumpe, el sistema continuará automáticamente desde donde se quedó.</small>
                </div>
                <?php endif; ?>



                <?php if ($countryStats): ?>
                <!-- Información de fuente de datos -->
                <?php if (isset($countryStats['source'])): ?>
                <div class="alert alert-info">
                    <i class="fas fa-database"></i> 
                    <?php if ($countryStats['source'] === 'database'): ?>
                        <strong>Datos desde la base de datos:</strong> Mostrando estadísticas completas desde la tabla geolocation_data
                    <?php elseif ($countryStats['source'] === 'combined'): ?>
                        <strong>Datos combinados:</strong> 
                        <?php echo number_format($countryStats['existingIPs'], 0, ',', '.'); ?> IPs históricas + 
                        <?php echo number_format($countryStats['newIPsProcessed'], 0, ',', '.'); ?> IPs nuevas procesadas
                    <?php endif; ?>
                </div>
                <?php endif; ?>
                
                <!-- Estadísticas generales -->
                <div class="row mb-4">
                    <div class="col-md-3">
                        <div class="stats-card">
                            <h4><i class="fas fa-users"></i> Total de IPs Únicas</h4>
                            <h2><?php echo number_format($countryStats['totalIPs'], 0, ',', '.'); ?></h2>
                            <small>IPs procesadas</small>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stats-card">
                            <h4><i class="fas fa-flag"></i> Países Detectados</h4>
                            <h2><?php echo count($countryStats['countries']); ?></h2>
                            <small>Países únicos</small>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stats-card">
                            <h4><i class="fas fa-map-marker-alt"></i> Ubicaciones</h4>
                            <h2 id="locations-count">-</h2>
                            <small>Con coordenadas</small>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stats-card">
                            <h4><i class="fas fa-clock"></i> Última Actualización</h4>
                            <h2><?php echo $lastUpdate ?: 'Nunca'; ?></h2>
                            <small>Datos en tiempo real</small>
                        </div>
                    </div>
                </div>

                <!-- Pestañas de estadísticas -->
                <ul class="nav nav-tabs stats-tabs" id="statsTabs" role="tablist">
                    <li class="nav-item" role="presentation">
                        <button class="nav-link active" id="countries-tab" data-bs-toggle="tab" data-bs-target="#countries" type="button" role="tab">
                            <i class="fas fa-flag"></i> Países
                        </button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="map-tab" data-bs-toggle="tab" data-bs-target="#map" type="button" role="tab">
                            <i class="fas fa-map"></i> Mapa Mundial
                        </button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="regions-tab" data-bs-toggle="tab" data-bs-target="#regions" type="button" role="tab">
                            <i class="fas fa-map-pin"></i> Regiones
                        </button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="cities-tab" data-bs-toggle="tab" data-bs-target="#cities" type="button" role="tab">
                            <i class="fas fa-city"></i> Ciudades
                        </button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="isps-tab" data-bs-toggle="tab" data-bs-target="#isps" type="button" role="tab">
                            <i class="fas fa-wifi"></i> ISPs
                        </button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="brazil-cities-tab" data-bs-toggle="tab" data-bs-target="#brazil-cities" type="button" role="tab">
                            <i class="fas fa-flag"></i> Brasil
                        </button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="argentina-cities-tab" data-bs-toggle="tab" data-bs-target="#argentina-cities" type="button" role="tab">
                            <i class="fas fa-flag"></i> Argentina
                        </button>
                    </li>
                </ul>

                <div class="tab-content" id="statsTabsContent">
                    <!-- Pestaña de Países -->
                    <div class="tab-pane fade show active" id="countries" role="tabpanel">
                        <!-- Tabla de ranking de países -->
                        <div class="card">
                            <div class="card-header">
                                <h5 class="card-title mb-0">
                                    <i class="fas fa-trophy"></i> Ranking de Países por Cantidad de Jugadores
                                </h5>
                            </div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-hover">
                                        <thead class="table-dark">
                                            <tr>
                                                <th width="50">#</th>
                                                <th width="60">Bandera</th>
                                                <th>País</th>
                                                <th>Código</th>
                                                <th>Jugadores</th>
                                                <th>Porcentaje</th>
                                                <th>Distribución</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php 
                                            $rank = 1;
                                            foreach ($countryStats['countries'] as $countryCode => $country): 
                                            ?>
                                            <tr>
                                                <td>
                                                    <span class="badge bg-primary"><?php echo $rank; ?></span>
                                                </td>
                                                <td>
                                                    <img src="https://flagcdn.com/24x18/<?php echo strtolower($countryCode); ?>.png" 
                                                         alt="<?php echo htmlspecialchars($country['country']); ?>"
                                                         class="country-flag"
                                                         onerror="this.style.display='none'">
                                                </td>
                                                <td>
                                                    <strong><?php echo htmlspecialchars($country['country']); ?></strong>
                                                </td>
                                                <td>
                                                    <span class="badge bg-secondary"><?php echo $country['countryCode']; ?></span>
                                                </td>
                                                <td>
                                                    <strong><?php echo number_format($country['count'], 0, ',', '.'); ?></strong>
                                                </td>
                                                <td>
                                                    <strong><?php echo $country['percentage']; ?>%</strong>
                                                </td>
                                                <td>
                                                    <div class="progress progress-bar-custom">
                                                        <div class="progress-bar bg-success" 
                                                             style="width: <?php echo $country['percentage']; ?>%">
                                                        </div>
                                                    </div>
                                                </td>
                                            </tr>
                                            <?php 
                                            $rank++;
                                            endforeach; 
                                            ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Pestaña del Mapa -->
                    <div class="tab-pane fade" id="map" role="tabpanel">
                        <div class="map-container">
                            <div class="d-flex justify-content-between align-items-center mb-3">
                                <h5><i class="fas fa-globe"></i> Mapa Mundial de Usuarios</h5>
                                <div>
                                    <button class="btn btn-sm btn-primary" onclick="window.GeolocationMapUtils.refreshMap()">
                                        <i class="fas fa-sync-alt"></i> Actualizar
                                    </button>
                                    <select class="form-select form-select-sm d-inline-block w-auto ms-2" onchange="window.GeolocationMapUtils.filterByCountry(this.value)">
                                        <option value="all">Todos los países</option>
                                        <?php foreach (array_slice($countryStats['countries'], 0, 10, true) as $countryCode => $country): ?>
                                        <option value="<?php echo $countryCode; ?>"><?php echo htmlspecialchars($country['country']); ?></option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                            </div>
                            <div id="geolocation-map"></div>
                            <div class="mt-3">
                                <small class="text-muted">
                                    <i class="fas fa-info-circle"></i> 
                                    Los círculos representan la concentración de usuarios. El tamaño y color indican la cantidad de usuarios en cada ubicación.
                                </small>
                            </div>
                        </div>
                    </div>

                    <!-- Pestaña de Regiones -->
                    <div class="tab-pane fade" id="regions" role="tabpanel">
                        <div id="regions-content">
                            <div class="text-center">
                                <i class="fas fa-spinner fa-spin"></i> Cargando estadísticas de regiones...
                            </div>
                        </div>
                    </div>

                    <!-- Pestaña de Ciudades -->
                    <div class="tab-pane fade" id="cities" role="tabpanel">
                        <div id="cities-content">
                            <div class="text-center">
                                <i class="fas fa-spinner fa-spin"></i> Cargando estadísticas de ciudades...
                            </div>
                        </div>
                    </div>

                    <!-- Pestaña de ISPs -->
                    <div class="tab-pane fade" id="isps" role="tabpanel">
                        <div id="isps-content">
                            <div class="text-center">
                                <i class="fas fa-spinner fa-spin"></i> Cargando estadísticas de ISPs...
                            </div>
                        </div>
                    </div>
                    
                    <!-- Pestaña de Ciudades Brasileñas -->
                    <div class="tab-pane fade" id="brazil-cities" role="tabpanel">
                        <div id="brazil-cities-content">
                            <div class="text-center">
                                <i class="fas fa-spinner fa-spin"></i> Cargando estadísticas de ciudades brasileñas...
                            </div>
                        </div>
                    </div>
                    
                    <!-- Pestaña de Ciudades Argentinas -->
                    <div class="tab-pane fade" id="argentina-cities" role="tabpanel">
                        <div id="argentina-cities-content">
                            <div class="text-center">
                                <i class="fas fa-spinner fa-spin"></i> Cargando estadísticas de ciudades argentinas...
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Información adicional -->
                <div class="row mt-4">
                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header">
                                <h6><i class="fas fa-info-circle"></i> Información de la API</h6>
                            </div>
                                                         <div class="card-body">
                                 <ul class="list-unstyled mb-0">
                                     <li><strong>Proveedor:</strong> ip-api.com</li>
                                     <li><strong>Rate Limit:</strong> 45 requests/minuto</li>
                                     <li><strong>Cache:</strong> Habilitado para optimizar consultas</li>
                                     <li><strong>Última actualización:</strong> <?php echo $lastUpdate ?: 'Nunca'; ?></li>
                                     <?php if (isset($_GET['today'])): ?>
                                     <li><strong>Modo:</strong> <span class="badge bg-info">Solo Hoy (Optimizado)</span></li>
                                     <?php endif; ?>
                                 </ul>
                             </div>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header">
                                <h6><i class="fas fa-chart-pie"></i> Top 5 Países</h6>
                            </div>
                            <div class="card-body">
                                <?php 
                                $top5 = array_slice($countryStats['countries'], 0, 5, true);
                                foreach ($top5 as $countryCode => $country): 
                                ?>
                                <div class="d-flex justify-content-between align-items-center mb-2">
                                    <span><?php echo htmlspecialchars($country['country']); ?></span>
                                    <span class="badge bg-primary"><?php echo $country['count']; ?></span>
                                </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </div>
                </div>

                <?php else: ?>
                <!-- Estado inicial -->
                <div class="text-center py-5">
                    <i class="fas fa-globe fa-3x text-muted mb-3"></i>
                    <h4>No hay datos de geolocalización</h4>
                    <p class="text-muted">Selecciona una opción para comenzar:</p>
                    <div class="d-flex justify-content-center gap-3">
                        <a href="?refresh=1" class="btn btn-success btn-lg">
                            <i class="fas fa-database"></i> Ver Estadísticas
                        </a>
                        <a href="?refresh=1&today=1" class="btn btn-info btn-lg">
                            <i class="fas fa-calendar-day"></i> Procesar Nuevas IPs
                        </a>
                    </div>
                                         <p class="text-muted mt-3">
                         <small><i class="fas fa-info-circle"></i> <strong>Sistema optimizado:</strong> Muestra estadísticas completas desde la base de datos y procesa solo las IPs nuevas de hoy bajo control manual.</small>
                     </p>
                     <div class="alert alert-info mt-3">
                         <small><i class="fas fa-shield-alt"></i> <strong>Optimizado para evitar errores de recursos:</strong> El sistema procesa en lotes de 45 IPs con límites de tiempo y memoria.</small>
                     </div>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"></script>
    <script src="https://unpkg.com/leaflet.markercluster@1.4.1/dist/leaflet.markercluster.js"></script>
    <script src="assets/js/geolocation-map.js"></script>
         <script>
         // Auto-refresh desactivado - El usuario controla manualmente el procesamiento
         // Para activar el auto-refresh, descomenta las líneas siguientes:
         
         /*
         // Auto-refresh cada 5 minutos si hay datos
         <?php if ($countryStats && !$isProcessing): ?>
         setTimeout(function() {
             if (confirm('¿Deseas actualizar los datos de geolocalización?')) {
                 window.location.href = '?refresh=1';
             }
         }, 300000); // 5 minutos
         <?php endif; ?>
         
         // Auto-refresh inteligente si hay procesamiento en curso
         <?php if (isset($_SESSION['geolocation_partial_data'])): ?>
         <?php 
         $partial = $_SESSION['geolocation_partial_data'];
         $timeSinceLastUpdate = time() - ($partial['timestamp'] ?? 0);
         $hasTimeout = isset($_SESSION['geolocation_partial_data']['timeout']);
         
         // Auto-refresh más agresivo para completar más rápido
         if ($hasTimeout) {
             // Si hay timeout, continuar inmediatamente
             $refreshDelay = 5000; // 5 segundos para timeout
         } elseif ($timeSinceLastUpdate > 20) {
             $refreshDelay = 3000; // 3 segundos si está "atascado"
         } elseif ($timeSinceLastUpdate > 10) {
             $refreshDelay = 10000; // 10 segundos si está procesando
         } else {
             $refreshDelay = 20000; // 20 segundos normal
         }
         ?>
         
         <?php if ($hasTimeout): ?>
         // Countdown para timeout
         let countdown = 5;
         const countdownElement = document.getElementById('countdown');
         const countdownInterval = setInterval(function() {
             countdown--;
             if (countdownElement) {
                 countdownElement.textContent = countdown;
             }
             if (countdown <= 0) {
                 clearInterval(countdownInterval);
             }
         }, 1000);
         <?php endif; ?>
         setTimeout(function() {
             console.log('Auto-refreshing geolocation processing... (delay: <?php echo $refreshDelay/1000; ?>s)');
             window.location.reload();
         }, <?php echo $refreshDelay; ?>);
         <?php endif; ?>
         */
         
         // Cargar estadísticas detalladas cuando se cambie de pestaña
         document.addEventListener('DOMContentLoaded', function() {
             const tabs = document.querySelectorAll('[data-bs-toggle="tab"]');
             
             tabs.forEach(tab => {
                 tab.addEventListener('shown.bs.tab', function(e) {
                     const target = e.target.getAttribute('data-bs-target');
                     
                     if (target === '#regions') {
                         loadRegionsStats();
                     } else if (target === '#cities') {
                         loadCitiesStats();
                     } else if (target === '#isps') {
                         loadISPsStats();
                     } else if (target === '#map') {
                         // Actualizar contador de ubicaciones
                         updateLocationsCount();
                     } else if (target === '#brazil-cities') {
                         loadBrazilCitiesStats();
                     } else if (target === '#argentina-cities') {
                         loadArgentinaCitiesStats();
                     }
                 });
             });
         });
         
         // Función para cargar estadísticas de regiones
         function loadRegionsStats() {
             fetch('geolocalizacion.php?action=get_detailed_stats')
                 .then(response => response.json())
                 .then(data => {
                     if (data.success && data.stats.regions) {
                         const content = document.getElementById('regions-content');
                         let html = '<div class="table-responsive"><table class="table table-hover">';
                         html += '<thead class="table-dark"><tr><th>#</th><th>Región</th><th>Usuarios</th><th>Porcentaje</th><th>Distribución</th></tr></thead><tbody>';
                         
                         data.stats.regions.forEach((region, index) => {
                             const percentage = ((region.count / data.stats.total) * 100).toFixed(1);
                             html += `<tr>
                                 <td><span class="badge bg-primary">${index + 1}</span></td>
                                 <td><strong>${region.region_name}</strong></td>
                                 <td><strong>${region.count.toLocaleString()}</strong></td>
                                 <td><strong>${percentage}%</strong></td>
                                 <td><div class="progress progress-bar-custom"><div class="progress-bar bg-info" style="width: ${percentage}%"></div></div></td>
                             </tr>`;
                         });
                         
                         html += '</tbody></table></div>';
                         content.innerHTML = html;
                     }
                 })
                 .catch(error => {
                     console.error('Error cargando estadísticas de regiones:', error);
                     document.getElementById('regions-content').innerHTML = '<div class="alert alert-danger">Error al cargar los datos</div>';
                 });
         }
         
         // Función para cargar estadísticas de ciudades
         function loadCitiesStats() {
             fetch('geolocalizacion.php?action=get_detailed_stats')
                 .then(response => response.json())
                 .then(data => {
                     if (data.success && data.stats.cities) {
                         const content = document.getElementById('cities-content');
                         let html = '<div class="table-responsive"><table class="table table-hover">';
                         html += '<thead class="table-dark"><tr><th>#</th><th>Ciudad</th><th>Usuarios</th><th>Porcentaje</th><th>Distribución</th></tr></thead><tbody>';
                         
                         data.stats.cities.forEach((city, index) => {
                             const percentage = ((city.count / data.stats.total) * 100).toFixed(1);
                             html += `<tr>
                                 <td><span class="badge bg-primary">${index + 1}</span></td>
                                 <td><strong>${city.city}</strong></td>
                                 <td><strong>${city.count.toLocaleString()}</strong></td>
                                 <td><strong>${percentage}%</strong></td>
                                 <td><div class="progress progress-bar-custom"><div class="progress-bar bg-success" style="width: ${percentage}%"></div></div></td>
                             </tr>`;
                         });
                         
                         html += '</tbody></table></div>';
                         content.innerHTML = html;
                     }
                 })
                 .catch(error => {
                     console.error('Error cargando estadísticas de ciudades:', error);
                     document.getElementById('cities-content').innerHTML = '<div class="alert alert-danger">Error al cargar los datos</div>';
                 });
         }
         
         // Función para cargar estadísticas de ISPs
         function loadISPsStats() {
             fetch('geolocalizacion.php?action=get_detailed_stats')
                 .then(response => response.json())
                 .then(data => {
                     if (data.success && data.stats.isps) {
                         const content = document.getElementById('isps-content');
                         let html = '<div class="table-responsive"><table class="table table-hover">';
                         html += '<thead class="table-dark"><tr><th>#</th><th>ISP</th><th>Usuarios</th><th>Porcentaje</th><th>Distribución</th></tr></thead><tbody>';
                         
                         data.stats.isps.forEach((isp, index) => {
                             const percentage = ((isp.count / data.stats.total) * 100).toFixed(1);
                             html += `<tr>
                                 <td><span class="badge bg-primary">${index + 1}</span></td>
                                 <td><strong>${isp.isp}</strong></td>
                                 <td><strong>${isp.count.toLocaleString()}</strong></td>
                                 <td><strong>${percentage}%</strong></td>
                                 <td><div class="progress progress-bar-custom"><div class="progress-bar bg-warning" style="width: ${percentage}%"></div></div></td>
                             </tr>`;
                         });
                         
                         html += '</tbody></table></div>';
                         content.innerHTML = html;
                     }
                 })
                 .catch(error => {
                     console.error('Error cargando estadísticas de ISPs:', error);
                     document.getElementById('isps-content').innerHTML = '<div class="alert alert-danger">Error al cargar los datos</div>';
                 });
         }
         
         // Función para actualizar el contador de ubicaciones
         function updateLocationsCount() {
             fetch('geolocalizacion.php?action=get_coordinates&limit=1')
                 .then(response => response.json())
                 .then(data => {
                     if (data.success) {
                         const countElement = document.getElementById('locations-count');
                         if (countElement) {
                             countElement.textContent = data.coordinates.length;
                         }
                     }
                 })
                 .catch(error => {
                     console.error('Error actualizando contador de ubicaciones:', error);
                 });
         }
         
         // Función para cargar estadísticas de ciudades brasileñas
         function loadBrazilCitiesStats() {
             fetch('geolocalizacion.php?action=get_detailed_stats')
                 .then(response => response.json())
                 .then(data => {
                     if (data.success && data.stats.brazil_cities) {
                         const content = document.getElementById('brazil-cities-content');
                         let html = '<div class="table-responsive"><table class="table table-hover">';
                         html += '<thead class="table-dark"><tr><th>#</th><th>Ciudad</th><th>Usuarios</th><th>Porcentaje</th><th>Distribución</th></tr></thead><tbody>';
                         
                         data.stats.brazil_cities.forEach((city, index) => {
                             const percentage = ((city.count / data.stats.total) * 100).toFixed(1);
                             html += `<tr>
                                 <td><span class="badge bg-primary">${index + 1}</span></td>
                                 <td><strong>${city.city}</strong></td>
                                 <td><strong>${city.count.toLocaleString()}</strong></td>
                                 <td><strong>${percentage}%</strong></td>
                                 <td><div class="progress progress-bar-custom"><div class="progress-bar bg-success" style="width: ${percentage}%"></div></div></td>
                             </tr>`;
                         });
                         
                         html += '</tbody></table></div>';
                         content.innerHTML = html;
                     } else {
                         document.getElementById('brazil-cities-content').innerHTML = '<div class="alert alert-info">No hay datos de ciudades brasileñas disponibles.</div>';
                     }
                 })
                 .catch(error => {
                     console.error('Error cargando estadísticas de ciudades brasileñas:', error);
                     document.getElementById('brazil-cities-content').innerHTML = '<div class="alert alert-danger">Error al cargar los datos</div>';
                 });
         }
         
         // Función para cargar estadísticas de ciudades argentinas
         function loadArgentinaCitiesStats() {
             fetch('geolocalizacion.php?action=get_detailed_stats')
                 .then(response => response.json())
                 .then(data => {
                     if (data.success && data.stats.argentina_cities) {
                         const content = document.getElementById('argentina-cities-content');
                         let html = '<div class="table-responsive"><table class="table table-hover">';
                         html += '<thead class="table-dark"><tr><th>#</th><th>Ciudad</th><th>Usuarios</th><th>Porcentaje</th><th>Distribución</th></tr></thead><tbody>';
                         
                         data.stats.argentina_cities.forEach((city, index) => {
                             const percentage = ((city.count / data.stats.total) * 100).toFixed(1);
                             html += `<tr>
                                 <td><span class="badge bg-primary">${index + 1}</span></td>
                                 <td><strong>${city.city}</strong></td>
                                 <td><strong>${city.count.toLocaleString()}</strong></td>
                                 <td><strong>${percentage}%</strong></td>
                                 <td><div class="progress progress-bar-custom"><div class="progress-bar bg-info" style="width: ${percentage}%"></div></div></td>
                             </tr>`;
                         });
                         
                         html += '</tbody></table></div>';
                         content.innerHTML = html;
                     } else {
                         document.getElementById('argentina-cities-content').innerHTML = '<div class="alert alert-info">No hay datos de ciudades argentinas disponibles.</div>';
                     }
                 })
                 .catch(error => {
                     console.error('Error cargando estadísticas de ciudades argentinas:', error);
                     document.getElementById('argentina-cities-content').innerHTML = '<div class="alert alert-danger">Error al cargar los datos</div>';
                 });
         }
     </script>
</body>
</html>
