<?php
session_start();
if(!isset($_SESSION['user_id']) || !isset($_SESSION['username']) || $_SESSION['username'] !== 'jade') {
    header("Location: index.php");
    exit();
}

require_once "config/database_singleton.php";

try {
    $db = getDB('jade');
    
    // Obtener filtros
    $fecha_inicio = isset($_GET['fecha_inicio']) ? $_GET['fecha_inicio'] : date('Y-m-d', strtotime('-30 days'));
    $fecha_fin = isset($_GET['fecha_fin']) ? $_GET['fecha_fin'] : date('Y-m-d');
    $metodo = isset($_GET['metodo']) ? $_GET['metodo'] : '';
    $personaje = isset($_GET['personaje']) ? $_GET['personaje'] : '';
    
    // Estadísticas generales separadas por método de pago
    $query_stats = "SELECT 
        metodo_pago,
        COUNT(*) as total_transacciones,
        SUM(precio) as total_ventas, -- precio ya está en la moneda correcta (USD o ARS)
        SUM(cantidad) as total_coins, -- cantidad ya tiene los coins directamente
        COUNT(DISTINCT personaje) as total_personajes
    FROM items_pagos 
    WHERE estado_pago = 'approved'
    AND DATE(fecha) BETWEEN :fecha_inicio AND :fecha_fin
    GROUP BY metodo_pago";
    
    $stmt_stats = $db->prepare($query_stats);
    $stmt_stats->execute([
        ':fecha_inicio' => $fecha_inicio,
        ':fecha_fin' => $fecha_fin
    ]);
    $stats = $stmt_stats->fetchAll(PDO::FETCH_ASSOC);
    
    // Inicializar totales
    $total_transacciones = 0;
    $total_personajes = 0;
    $total_coins = 0;
    
    // Estadísticas por día separadas por método de pago
    $query_diario = "SELECT 
        DATE(fecha) as dia,
        metodo_pago,
        COUNT(*) as transacciones,
        SUM(precio) as total, -- precio es la cantidad de coins
        SUM(CASE 
            WHEN metodo_pago = 'paypal' THEN precio * 0.10 -- En USD (0.10 por coin)
            ELSE precio * 100 -- En ARS (100 por coin)
        END) as total_moneda
    FROM items_pagos 
    WHERE estado_pago = 'approved'
    AND DATE(fecha) BETWEEN :fecha_inicio AND :fecha_fin
    GROUP BY DATE(fecha), metodo_pago
    ORDER BY dia DESC, metodo_pago";
    
    $stmt_diario = $db->prepare($query_diario);
    $stmt_diario->execute([
        ':fecha_inicio' => $fecha_inicio,
        ':fecha_fin' => $fecha_fin
    ]);
    $stats_diario = $stmt_diario->fetchAll(PDO::FETCH_ASSOC);
    
    // Top 3 meses con más ganancias por método de pago
    $query_top_meses = "SELECT 
        metodo_pago,
        DATE_FORMAT(fecha, '%Y-%m') as mes,
        DATE_FORMAT(fecha, '%M %Y') as mes_nombre,
        SUM(precio) as total_ventas,
        COUNT(*) as total_transacciones
    FROM items_pagos 
    WHERE estado_pago = 'approved'
    GROUP BY metodo_pago, DATE_FORMAT(fecha, '%Y-%m')
    HAVING metodo_pago IN ('paypal', 'mercadopago')
    ORDER BY metodo_pago, total_ventas DESC";

    $stmt_top_meses = $db->query($query_top_meses);
    $top_meses = [];
    $current_method = null;
    $count = 0;

    while ($row = $stmt_top_meses->fetch(PDO::FETCH_ASSOC)) {
        if ($current_method !== $row['metodo_pago']) {
            $current_method = $row['metodo_pago'];
            $count = 0;
        }
        if ($count < 3) {
            $top_meses[$row['metodo_pago']][] = $row;
            $count++;
        }
    }

    // Obtener métodos de pago únicos para el filtro
    $query_metodos = "SELECT DISTINCT metodo_pago FROM items_pagos ORDER BY metodo_pago";
    $stmt_metodos = $db->query($query_metodos);
    $metodos_pago = $stmt_metodos->fetchAll(PDO::FETCH_COLUMN);
    
    // Obtener personajes únicos para el filtro
    $query_personajes = "SELECT DISTINCT personaje FROM items_pagos ORDER BY personaje";
    $stmt_personajes = $db->query($query_personajes);
    $personajes = $stmt_personajes->fetchAll(PDO::FETCH_COLUMN);
    
    // Construir consulta para transacciones con filtros
    $where_clauses = ["1=1"];
    $params = [];
    
    if ($metodo) {
        $where_clauses[] = "metodo_pago = :metodo";
        $params[':metodo'] = $metodo;
    }
    if ($personaje) {
        $where_clauses[] = "personaje = :personaje";
        $params[':personaje'] = $personaje;
    }
    if ($fecha_inicio && $fecha_fin) {
        $where_clauses[] = "DATE(fecha) BETWEEN :fecha_inicio AND :fecha_fin";
        $params[':fecha_inicio'] = $fecha_inicio;
        $params[':fecha_fin'] = $fecha_fin;
    }
    
    $where_sql = implode(" AND ", $where_clauses);
    
    // Obtener transacciones
    $query_trans = "SELECT * FROM items_pagos WHERE $where_sql ORDER BY fecha DESC LIMIT 1000";
    $stmt_trans = $db->prepare($query_trans);
    $stmt_trans->execute($params);
    $transacciones = $stmt_trans->fetchAll(PDO::FETCH_ASSOC);
    
} catch(PDOException $e) {
    $error = "Error al obtener datos: " . $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Panel de Donaciones</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.datatables.net/1.11.5/css/dataTables.bootstrap5.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/daterangepicker/daterangepicker.css" rel="stylesheet">
    <link href="assets/css/dark-theme.css" rel="stylesheet">
    <style>
        .stat-card {
            background-color: #2d2d2d;
            border: 1px solid #404040;
            border-radius: 12px;
            padding: 24px;
            margin-bottom: 20px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.3);
            transition: all 0.3s ease;
            color: #e0e0e0;
        }
        .stat-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 24px rgba(0,0,0,0.4);
            border-color: #505050;
        }
        .stat-card h5 {
            color: #b0b0b0;
            font-weight: 600;
            margin-bottom: 12px;
        }
        .stat-card h2 {
            color: #ffffff;
            font-weight: 700;
            margin-bottom: 8px;
        }
        .chart-container {
            background-color: #2d2d2d;
            border: 1px solid #404040;
            border-radius: 12px;
            padding: 24px;
            margin-bottom: 20px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.3);
            color: #e0e0e0;
        }
        .chart-container h5 {
            color: #ffffff;
            font-weight: 600;
            margin-bottom: 20px;
        }
        /* DataTables dark theme */
        .dataTables_wrapper .dataTables_length,
        .dataTables_wrapper .dataTables_filter,
        .dataTables_wrapper .dataTables_info,
        .dataTables_wrapper .dataTables_processing,
        .dataTables_wrapper .dataTables_paginate {
            color: #e0e0e0;
        }
        .dataTables_wrapper .dataTables_paginate .paginate_button {
            color: #e0e0e0 !important;
            background-color: #2d2d2d !important;
            border-color: #404040 !important;
        }
        .dataTables_wrapper .dataTables_paginate .paginate_button:hover {
            background-color: #404040 !important;
            border-color: #505050 !important;
            color: #ffffff !important;
        }
        .dataTables_wrapper .dataTables_paginate .paginate_button.current {
            background-color: #0d6efd !important;
            border-color: #0d6efd !important;
            color: #ffffff !important;
        }
        /* DateRangePicker dark theme */
        .daterangepicker {
            background-color: #2d2d2d;
            border: 1px solid #404040;
            color: #e0e0e0;
        }
        .daterangepicker .ranges li {
            color: #e0e0e0;
        }
        .daterangepicker .ranges li:hover {
            background-color: #404040;
        }
        .daterangepicker .ranges li.active {
            background-color: #0d6efd;
        }
        .daterangepicker .calendar-table {
            background-color: #2d2d2d;
            border-color: #404040;
        }
        .daterangepicker .calendar-table .next span,
        .daterangepicker .calendar-table .prev span {
            border-color: #e0e0e0;
        }
        .daterangepicker .calendar-table .table-condensed td,
        .daterangepicker .calendar-table .table-condensed th {
            color: #e0e0e0;
        }
        .daterangepicker .calendar-table .table-condensed td.available:hover {
            background-color: #404040;
        }
        .daterangepicker .calendar-table .table-condensed td.active {
            background-color: #0d6efd;
        }
        .daterangepicker .calendar-table .table-condensed td.in-range {
            background-color: #404040;
            color: #ffffff;
        }
        .daterangepicker td.off, .daterangepicker td.off.in-range, .daterangepicker td.off.start-date, .daterangepicker td.off.end-date {
            background-color: #2d2d2d;
            border-color: transparent;
            color: #666;
        }
        
        /* Tablas con tema oscuro */
        .table {
            color: #b0b0b0;
            --bs-table-striped-color: #ffffff;
        }
        .table th {
            color: #ffffff;
            background-color: #353535;
            border-color: #404040;
        }
        .table td {
            color: #b0b0b0;
            border-color: #404040;
        }
        .table-striped > tbody > tr:nth-of-type(odd) > td {
            background-color: #2d2d2d;
        }
        .table-striped > tbody > tr:nth-of-type(even) > td {
            background-color: #353535;
        }
        .table-sm th,
        .table-sm td {
            color: #b0b0b0;
        }
        
        /* Cards de Bootstrap con tema oscuro */
        .card {
            background-color: #2d2d2d;
            border: 1px solid #404040;
            color: #e0e0e0;
        }
        .card-body {
            color: #e0e0e0;
        }
        .card-title {
            color: #ffffff;
        }
        
        /* Formularios con tema oscuro */
        .form-label {
            color: #b0b0b0;
        }
        .form-select {
            background-color: #2d2d2d;
            border: 1px solid #404040;
            color: #e0e0e0;
        }
        .form-select:focus {
            background-color: #353535;
            border-color: #0d6efd;
            color: #ffffff;
            box-shadow: 0 0 0 0.2rem rgba(13, 110, 253, 0.25);
        }
        .form-select option {
            background-color: #2d2d2d;
            color: #e0e0e0;
        }
        
        /* DataTables específicos */
        .dataTables_wrapper .dataTables_length select,
        .dataTables_wrapper .dataTables_filter input {
            background-color: #2d2d2d;
            border: 1px solid #404040;
            color: #e0e0e0;
        }
        .dataTables_wrapper .dataTables_length select:focus,
        .dataTables_wrapper .dataTables_filter input:focus {
            background-color: #353535;
            border-color: #0d6efd;
            color: #ffffff;
        }
        .dataTables_wrapper .dataTables_length select option {
            background-color: #2d2d2d;
            color: #e0e0e0;
        }
        
        /* Texto muted con tema oscuro */
        .text-muted {
            color: #b0b0b0 !important;
        }
        
        /* Títulos principales */
        h1, h2, h3, h4, h5, h6 {
            color: #ffffff;
        }
        
        /* Contenedor principal */
        .container-fluid {
            background-color: #1a1a1a;
        }
    </style>
</head>
<body>
    <div class="container-fluid py-4">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h1 class="mb-0">Panel de Donaciones</h1>
            <a href="dashboard.php" class="btn btn-secondary">
                <i class="fas fa-arrow-left"></i> Volver al Dashboard
            </a>
        </div>
        
        <?php if (isset($error)): ?>
            <div class="alert alert-danger"><?php echo $error; ?></div>
        <?php endif; ?>

        <!-- Filtros -->
        <div class="card mb-4">
            <div class="card-body">
                <form method="GET" class="row g-3">
                    <div class="col-md-4">
                        <label class="form-label">Rango de Fechas</label>
                        <input type="text" class="form-control" id="daterange" name="daterange" 
                               value="<?php echo date('d/m/Y', strtotime($fecha_inicio)) . ' - ' . date('d/m/Y', strtotime($fecha_fin)); ?>">
                        <input type="hidden" name="fecha_inicio" id="fecha_inicio" value="<?php echo $fecha_inicio; ?>">
                        <input type="hidden" name="fecha_fin" id="fecha_fin" value="<?php echo $fecha_fin; ?>">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Método de Pago</label>
                        <select class="form-select" name="metodo">
                            <option value="">Todos</option>
                            <?php foreach ($metodos_pago as $m): ?>
                                <option value="<?php echo htmlspecialchars($m); ?>" <?php echo $metodo == $m ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($m); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Personaje</label>
                        <select class="form-select" name="personaje">
                            <option value="">Todos</option>
                            <?php foreach ($personajes as $p): ?>
                                <option value="<?php echo htmlspecialchars($p); ?>" <?php echo $personaje == $p ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($p); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">&nbsp;</label>
                        <button type="submit" class="btn btn-primary w-100">Filtrar</button>
                    </div>
                </form>
            </div>
        </div>

        <!-- Tarjetas de Resumen -->
        <div class="row mb-4">
            <?php 
            // Calcular totales
            foreach ($stats as $stat) {
                $total_transacciones += $stat['total_transacciones'];
                $total_personajes += $stat['total_personajes'];
                $total_coins += $stat['total_coins'];
            }

            // Mostrar estadísticas por método de pago
            foreach ($stats as $stat): 
            ?>
            <div class="col-md-3">
                <div class="stat-card">
                    <h5 class="card-title">Total Ventas <?php echo $stat['metodo_pago'] == 'paypal' ? '(USD)' : '(ARS)'; ?></h5>
                    <h2 class="mb-0">
                        <?php 
                        if ($stat['metodo_pago'] == 'paypal') {
                            echo 'USD ' . number_format($stat['total_ventas'], 2); // En USD (ya calculado)
                        } else {
                            echo '$' . number_format($stat['total_ventas'], 0); // En ARS directo
                        }
                        ?>
                    </h2>
                    <small class="text-muted">
                        <?php echo number_format($stat['total_coins'], 0); ?> coins vía <?php echo ucfirst($stat['metodo_pago']); ?>
                    </small>
                </div>
            </div>
            <?php endforeach; ?>

            <!-- Total Transacciones -->
            <div class="col-md-3">
                <div class="stat-card">
                    <h5 class="card-title">Total Transacciones</h5>
                    <h2 class="mb-0"><?php echo number_format($total_transacciones); ?></h2>
                    <div class="mt-2">
                        <?php foreach ($stats as $stat): ?>
                            <small class="d-block text-muted">
                                <?php echo ucfirst($stat['metodo_pago']); ?>: <?php echo number_format($stat['total_transacciones']); ?> transacciones
                            </small>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>

            <!-- Total Personajes -->
            <div class="col-md-3">
                <div class="stat-card">
                    <h5 class="card-title">Personajes Únicos</h5>
                    <h2 class="mb-0"><?php echo number_format($total_personajes); ?></h2>
                    <small class="text-muted">Que realizaron compras</small>
                </div>
            </div>

            <!-- Total Coins -->
            <div class="col-md-3">
                <div class="stat-card">
                    <h5 class="card-title">Total Coins Generados</h5>
                    <h2 class="mb-0"><?php echo number_format($total_coins, 0); ?></h2>
                    <small class="text-muted">Coins totales entregados</small>
                </div>
            </div>
        </div>

        <!-- Top 3 Meses -->
        <div class="row mb-4">
            <?php foreach (['paypal', 'mercadopago'] as $metodo): ?>
            <div class="col-md-6">
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title">Top 3 Meses - <?php echo $metodo == 'paypal' ? 'PayPal (USD)' : 'MercadoPago (ARS)'; ?></h5>
                        <div class="table-responsive">
                            <table class="table table-sm">
                                <thead>
                                    <tr>
                                        <th>Mes</th>
                                        <th>Total</th>
                                        <th>Transacciones</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (isset($top_meses[$metodo])): ?>
                                        <?php foreach ($top_meses[$metodo] as $mes): ?>
                                            <tr>
                                                <td><?php echo ucfirst($mes['mes_nombre']); ?></td>
                                                <td>
                                                    <?php if ($metodo == 'paypal'): ?>
                                                        USD <?php echo number_format($mes['total_ventas'], 2); ?>
                                                    <?php else: ?>
                                                        $<?php echo number_format($mes['total_ventas'], 0); ?>
                                                    <?php endif; ?>
                                                </td>
                                                <td><?php echo number_format($mes['total_transacciones']); ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php else: ?>
                                        <tr>
                                            <td colspan="3" class="text-center">No hay datos disponibles</td>
                                        </tr>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
            <?php endforeach; ?>
        </div>

        <!-- Gráfico -->
        <div class="row mb-4">
            <div class="col-12">
                <div class="chart-container">
                    <h5 class="card-title">Tendencia de Ventas</h5>
                    <canvas id="ventasChart"></canvas>
                </div>
            </div>
        </div>

        <!-- Tabla de Transacciones -->
        <div class="card">
            <div class="card-body">
                <h5 class="card-title mb-4">Transacciones</h5>
                <div class="table-responsive">
                    <table id="transaccionesTable" class="table table-striped">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Pay ID</th>
                                <th>Personaje</th>
                                <th>Método</th>
                                <th>Estado</th>
                                <th>Precio</th>
                                <th>Coins</th>
                                <th>Fecha</th>
                                <th>Reclamado</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($transacciones as $t): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($t['id']); ?></td>
                                <td><?php echo htmlspecialchars($t['pay_id']); ?></td>
                                <td><?php echo htmlspecialchars($t['personaje']); ?></td>
                                <td><?php echo htmlspecialchars($t['metodo_pago']); ?></td>
                                <td>
                                    <span class="badge bg-<?php echo $t['estado_pago'] == 'approved' ? 'success' : 'warning'; ?>">
                                        <?php echo htmlspecialchars($t['estado_pago']); ?>
                                    </span>
                                </td>
                                <td><?php echo number_format($t['precio'], 0); ?></td>
                                <td><?php echo number_format($t['cantidad'], 0); ?></td>
                                <td><?php echo date('Y-m-d H:i', strtotime($t['fecha'])); ?></td>
                                <td>
                                    <span class="badge bg-<?php echo $t['is_claimed'] ? 'success' : 'warning'; ?>">
                                        <?php echo $t['is_claimed'] ? 'Sí' : 'No'; ?>
                                    </span>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.datatables.net/1.11.5/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.11.5/js/dataTables.bootstrap5.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/moment/moment.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/daterangepicker/daterangepicker.min.js"></script>
    <script>
        $(document).ready(function() {
            // Inicializar DataTable
            $('#transaccionesTable').DataTable({
                "language": {
                    "url": "//cdn.datatables.net/plug-ins/1.11.5/i18n/es-ES.json"
                },
                "order": [[8, "desc"]], // Ordenar por fecha descendente
                "pageLength": 25
            });

            // Inicializar DateRangePicker
            $('#daterange').daterangepicker({
                locale: {
                    format: 'DD/MM/YYYY',
                    applyLabel: 'Aplicar',
                    cancelLabel: 'Cancelar',
                    fromLabel: 'Desde',
                    toLabel: 'Hasta',
                    customRangeLabel: 'Rango Personalizado',
                    daysOfWeek: ['Do', 'Lu', 'Ma', 'Mi', 'Ju', 'Vi', 'Sa'],
                    monthNames: ['Enero', 'Febrero', 'Marzo', 'Abril', 'Mayo', 'Junio', 'Julio', 'Agosto', 'Septiembre', 'Octubre', 'Noviembre', 'Diciembre']
                },
                startDate: moment('<?php echo $fecha_inicio; ?>'),
                endDate: moment('<?php echo $fecha_fin; ?>')
            }, function(start, end) {
                $('#fecha_inicio').val(start.format('YYYY-MM-DD'));
                $('#fecha_fin').val(end.format('YYYY-MM-DD'));
            });
        });

        // Preparar datos para el gráfico
        <?php
        $dias = array_unique(array_map(function($row) {
            return $row['dia'];
        }, $stats_diario));
        
        $datos_paypal = [];
        $datos_mercadopago = [];
        
        foreach ($dias as $dia) {
            $total_paypal = 0;
            $total_mercadopago = 0;
            
            foreach ($stats_diario as $stat) {
                if ($stat['dia'] == $dia) {
                    if ($stat['metodo_pago'] == 'paypal') {
                        $total_paypal = $stat['total'];
                    } else if ($stat['metodo_pago'] == 'mercadopago') {
                        $total_mercadopago = $stat['total'];
                    }
                }
            }
            
            $datos_paypal[] = $total_paypal;
            $datos_mercadopago[] = $total_mercadopago;
        }
        ?>
        
        // Configurar gráfico de ventas
        const ctx = document.getElementById('ventasChart').getContext('2d');
        const ventasChart = new Chart(ctx, {
            type: 'line',
            data: {
                labels: <?php echo json_encode(array_map(function($dia) { 
                    return date('d/m/Y', strtotime($dia)); 
                }, array_reverse($dias))); ?>,
                datasets: [{
                    label: 'PayPal (USD)',
                    data: <?php echo json_encode(array_reverse($datos_paypal)); ?>,
                    borderColor: 'rgb(0, 122, 255)',
                    tension: 0.1,
                    fill: false
                },
                {
                    label: 'MercadoPago (ARS)',
                    data: <?php echo json_encode(array_reverse($datos_mercadopago)); ?>,
                    borderColor: 'rgb(255, 122, 0)',
                    tension: 0.1,
                    fill: false
                }]
            },
            options: {
                responsive: true,
                plugins: {
                    legend: {
                        position: 'top',
                    },
                    title: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            callback: function(value) {
                                return '$' + Math.round(value).toLocaleString();
                            }
                        }
                    }
                }
            }
        });
    </script>
</body>
</html>