<?php
/**
 * Script de Diagnóstico Detallado de Conexiones
 * Analiza todas las conexiones abiertas y muestra información específica
 */

session_start();

// Verificar si el usuario está logueado
if (!isset($_SESSION['user_id'])) {
    header("Location: index.php");
    exit();
}

require_once "config/database_advanced.php";
require_once "includes/log_viewer.php";

// Inicializar el gestor de base de datos avanzado
$dbManager = AdvancedDatabaseManager::getInstance();

// Obtener estadísticas actuales
$stats = $dbManager->getAdvancedStats();
$health = $dbManager->checkConnectionHealth();

// Inicializar el visualizador de logs
$logViewer = new LogViewer();
$logs = $logViewer->loadLogs();

// Analizar logs para detectar conexiones abiertas
$openedConnections = 0;
$closedConnections = 0;
$openConnections = [];
$connectionByPage = [];

foreach ($logs as $log) {
    if ($log['action'] === 'OPENING') {
        $openedConnections++;
        $openConnections[] = [
            'type' => $log['type'],
            'timestamp' => $log['timestamp'],
            'message' => $log['message']
        ];
        
        // Intentar identificar la página basándose en el timestamp
        $connectionByPage[] = [
            'timestamp' => $log['timestamp'],
            'type' => $log['type'],
            'estimated_page' => 'Desconocida'
        ];
    } elseif ($log['action'] === 'CLOSING') {
        $closedConnections++;
    }
}

// Obtener información de memoria
$memoryUsage = memory_get_usage(true);
$memoryPeak = memory_get_peak_usage(true);

// Probar conexiones activas
$activeConnections = [];
try {
    $crmConnection = $dbManager->getCRMConnection();
    $activeConnections['crm'] = true;
} catch (Exception $e) {
    $activeConnections['crm'] = false;
}

try {
    $jadeConnection = $dbManager->getJadeConnection();
    $activeConnections['jade'] = true;
} catch (Exception $e) {
    $activeConnections['jade'] = false;
}

// Obtener información del sistema
$systemInfo = [
    'php_version' => PHP_VERSION,
    'memory_limit' => ini_get('memory_limit'),
    'max_execution_time' => ini_get('max_execution_time'),
    'server_time' => date('Y-m-d H:i:s'),
    'uptime' => function_exists('sys_getloadavg') ? sys_getloadavg() : 'N/A'
];

?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Diagnóstico Detallado - Panel de Estadísticas</title>
    <link rel="stylesheet" href="assets/css/dark-theme.css">
    <link rel="stylesheet" href="assets/css/monitoring.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
</head>
<body>
    <div class="monitoring-container">
        <!-- Header -->
        <div class="monitoring-header">
            <h1 class="monitoring-title">
                <i class="fas fa-search"></i> Diagnóstico Detallado de Conexiones
            </h1>
            <p class="monitoring-subtitle">
                Análisis completo del problema de conexiones acumuladas
            </p>
            <div style="text-align: center; margin-top: 15px;">
                <a href="herramientas_monitoreo.php" class="btn-monitoring btn-primary">
                    <i class="fas fa-arrow-left"></i> Volver a Herramientas
                </a>
            </div>
        </div>

        <!-- Estado Crítico -->
        <div class="monitoring-card" style="border: 2px solid #e74c3c; background: linear-gradient(45deg, #e74c3c, #c0392b);">
            <div class="card-header">
                <div class="card-icon" style="background: linear-gradient(45deg, #e74c3c, #c0392b);">
                    <i class="fas fa-exclamation-triangle"></i>
                </div>
                <h3 class="card-title" style="color: white;">ESTADO CRÍTICO DETECTADO</h3>
            </div>
            
            <div style="margin-bottom: 20px; color: white;">
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                    <span><strong>Conexiones Abiertas:</strong></span>
                    <span style="font-size: 1.2em; font-weight: bold;"><?php echo $openedConnections; ?></span>
                </div>
                
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                    <span><strong>Conexiones Cerradas:</strong></span>
                    <span style="font-size: 1.2em; font-weight: bold;"><?php echo $closedConnections; ?></span>
                </div>
                
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                    <span><strong>CONEXIONES HUÉRFANAS:</strong></span>
                    <span style="font-size: 1.5em; font-weight: bold; color: #f1c40f;"><?php echo $openedConnections - $closedConnections; ?></span>
                </div>
            </div>
            
            <div style="text-align: center;">
                <a href="limpieza_emergencia.php" class="btn-monitoring btn-danger" style="background: #c0392b; border-color: #c0392b;">
                    <i class="fas fa-fire-extinguisher"></i> LIMPIEZA DE EMERGENCIA
                </a>
            </div>
        </div>

        <!-- Estado Actual -->
        <div class="monitoring-grid">
            <div class="monitoring-card">
                <div class="card-header">
                    <div class="card-icon" style="background: linear-gradient(45deg, #667eea, #764ba2);">
                        <i class="fas fa-database"></i>
                    </div>
                    <h3 class="card-title">Estado Actual</h3>
                </div>
                
                <div style="margin-bottom: 20px;">
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Conexión CRM:</strong></span>
                        <span class="status-indicator <?php echo $activeConnections['crm'] ? 'status-active' : 'status-inactive'; ?>">
                            <?php echo $activeConnections['crm'] ? '🔴 ABIERTA' : '✅ CERRADA'; ?>
                        </span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Conexión Jade:</strong></span>
                        <span class="status-indicator <?php echo $activeConnections['jade'] ? 'status-active' : 'status-inactive'; ?>">
                            <?php echo $activeConnections['jade'] ? '🔴 ABIERTA' : '✅ CERRADA'; ?>
                        </span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Total Conexiones:</strong></span>
                        <span class="status-indicator status-active">
                            <?php echo array_sum($activeConnections); ?>
                        </span>
                    </div>
                </div>
            </div>

            <div class="monitoring-card">
                <div class="card-header">
                    <div class="card-icon" style="background: linear-gradient(45deg, #00b894, #00cec9);">
                        <i class="fas fa-chart-bar"></i>
                    </div>
                    <h3 class="card-title">Análisis de Logs</h3>
                </div>
                
                <div style="margin-bottom: 20px;">
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Conexiones Abiertas:</strong></span>
                        <span class="status-indicator status-warning">
                            <?php echo $openedConnections; ?>
                        </span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Conexiones Cerradas:</strong></span>
                        <span class="status-indicator status-active">
                            <?php echo $closedConnections; ?>
                        </span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Diferencia:</strong></span>
                        <span class="status-indicator <?php echo ($openedConnections - $closedConnections) > 0 ? 'status-inactive' : 'status-active'; ?>">
                            <?php echo $openedConnections - $closedConnections; ?>
                        </span>
                    </div>
                </div>
            </div>

            <div class="monitoring-card">
                <div class="card-header">
                    <div class="card-icon" style="background: linear-gradient(45deg, #fdcb6e, #e17055);">
                        <i class="fas fa-memory"></i>
                    </div>
                    <h3 class="card-title">Uso de Memoria</h3>
                </div>
                
                <div style="margin-bottom: 20px;">
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Memoria Actual:</strong></span>
                        <span><?php echo number_format($memoryUsage / 1024 / 1024, 2); ?> MB</span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Memoria Pico:</strong></span>
                        <span><?php echo number_format($memoryPeak / 1024 / 1024, 2); ?> MB</span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Límite de Memoria:</strong></span>
                        <span><?php echo $systemInfo['memory_limit']; ?></span>
                    </div>
                </div>
            </div>

            <div class="monitoring-card">
                <div class="card-header">
                    <div class="card-icon" style="background: linear-gradient(45deg, #e17055, #d63031);">
                        <i class="fas fa-cogs"></i>
                    </div>
                    <h3 class="card-title">Acciones</h3>
                </div>
                
                <div style="margin-bottom: 20px;">
                    <p style="color: #7f8c8d; font-size: 0.9rem; margin-bottom: 15px;">
                        Acciones para resolver el problema de conexiones
                    </p>
                </div>
                
                <div style="display: flex; flex-direction: column; gap: 10px;">
                    <a href="limpieza_emergencia.php" class="btn-monitoring btn-danger">
                        <i class="fas fa-fire-extinguisher"></i> Limpieza de Emergencia
                    </a>
                    
                    <a href="correccion_conexiones.php" class="btn-monitoring btn-warning">
                        <i class="fas fa-wrench"></i> Corregir Sistema
                    </a>
                    
                    <button class="btn-monitoring btn-success" onclick="location.reload()">
                        <i class="fas fa-sync-alt"></i> Actualizar Diagnóstico
                    </button>
                </div>
            </div>
        </div>

        <!-- Últimas Conexiones Abiertas -->
        <?php if (!empty($openConnections)): ?>
        <div class="monitoring-card">
            <div class="card-header">
                <div class="card-icon" style="background: linear-gradient(45deg, #e17055, #d63031);">
                    <i class="fas fa-exclamation-triangle"></i>
                </div>
                <h3 class="card-title">Últimas Conexiones Abiertas (Sin Cerrar)</h3>
            </div>
            
            <div style="overflow-x: auto;">
                <table class="logs-table">
                    <thead>
                        <tr>
                            <th>Timestamp</th>
                            <th>Tipo</th>
                            <th>Mensaje</th>
                            <th>Estado</th>
                            <th>Tiempo Abierta</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach (array_slice($openConnections, 0, 15) as $connection): ?>
                        <tr>
                            <td>
                                <strong><?php echo date('H:i:s', strtotime($connection['timestamp'])); ?></strong><br>
                                <small><?php echo date('d/m/Y', strtotime($connection['timestamp'])); ?></small>
                            </td>
                            <td>
                                <span class="status-indicator <?php echo strtolower($connection['type']) === 'crm' ? 'status-active' : 'status-warning'; ?>">
                                    <?php echo $connection['type']; ?>
                                </span>
                            </td>
                            <td><?php echo htmlspecialchars($connection['message']); ?></td>
                            <td>
                                <span class="status-indicator status-inactive">
                                    🔴 ABIERTA
                                </span>
                            </td>
                            <td>
                                <?php 
                                $timeDiff = time() - strtotime($connection['timestamp']);
                                if ($timeDiff < 60) {
                                    echo $timeDiff . 's';
                                } elseif ($timeDiff < 3600) {
                                    echo floor($timeDiff / 60) . 'm ' . ($timeDiff % 60) . 's';
                                } else {
                                    echo floor($timeDiff / 3600) . 'h ' . floor(($timeDiff % 3600) / 60) . 'm';
                                }
                                ?>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        <?php endif; ?>

        <!-- Información del Sistema -->
        <div class="monitoring-card">
            <div class="card-header">
                <div class="card-icon" style="background: linear-gradient(45deg, #6c5ce7, #a29bfe);">
                    <i class="fas fa-info-circle"></i>
                </div>
                <h3 class="card-title">Información del Sistema</h3>
            </div>
            
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px;">
                <div>
                    <strong>PHP Version:</strong> <?php echo $systemInfo['php_version']; ?>
                </div>
                <div>
                    <strong>Memory Limit:</strong> <?php echo $systemInfo['memory_limit']; ?>
                </div>
                <div>
                    <strong>Max Execution Time:</strong> <?php echo $systemInfo['max_execution_time']; ?>s
                </div>
                <div>
                    <strong>Server Time:</strong> <?php echo $systemInfo['server_time']; ?>
                </div>
                <?php if ($systemInfo['uptime'] !== 'N/A'): ?>
                <div>
                    <strong>System Load:</strong> <?php echo implode(', ', array_map('number_format', $systemInfo['uptime'])); ?>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <script>
    // Auto-refresh cada 10 segundos para monitoreo en tiempo real
    setInterval(function() {
        location.reload();
    }, 10000);
    </script>
</body>
</html>
