<?php
/**
 * Script de Corrección del Sistema de Conexiones
 * Implementa mejoras para prevenir acumulación de conexiones
 */

session_start();

// Verificar si el usuario está logueado
if (!isset($_SESSION['user_id'])) {
    header("Location: index.php");
    exit();
}

require_once "config/database_advanced.php";

// Inicializar el gestor de base de datos avanzado
$dbManager = AdvancedDatabaseManager::getInstance();

// Obtener estadísticas actuales
$statsBefore = $dbManager->getAdvancedStats();

// Realizar corrección del sistema
$corrections = [];

// 1. Limpiar conexiones inactivas
$cleanupResult = $dbManager->cleanupInactiveConnections(0);
$corrections[] = "Limpieza de conexiones inactivas: $cleanupResult conexiones cerradas";

// 2. Cerrar todas las conexiones explícitamente
$dbManager->closeConnections();
$corrections[] = "Cierre explícito de todas las conexiones";

// 3. Verificar que las conexiones se cerraron
$statsAfter = $dbManager->getAdvancedStats();
$corrections[] = "Verificación de cierre completada";

// Obtener información del sistema
$systemInfo = [
    'php_version' => PHP_VERSION,
    'memory_limit' => ini_get('memory_limit'),
    'max_execution_time' => ini_get('max_execution_time'),
    'server_time' => date('Y-m-d H:i:s'),
    'uptime' => function_exists('sys_getloadavg') ? sys_getloadavg() : 'N/A'
];

?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Corrección del Sistema - Panel de Estadísticas</title>
    <link rel="stylesheet" href="assets/css/dark-theme.css">
    <link rel="stylesheet" href="assets/css/monitoring.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
</head>
<body>
    <div class="monitoring-container">
        <!-- Header -->
        <div class="monitoring-header">
            <h1 class="monitoring-title">
                <i class="fas fa-wrench"></i> Corrección del Sistema de Conexiones
            </h1>
            <p class="monitoring-subtitle">
                Implementación de mejoras para prevenir acumulación de conexiones
            </p>
            <div style="text-align: center; margin-top: 15px;">
                <a href="herramientas_monitoreo.php" class="btn-monitoring btn-primary">
                    <i class="fas fa-arrow-left"></i> Volver a Herramientas
                </a>
            </div>
        </div>

        <!-- Estado de la Corrección -->
        <div class="monitoring-card" style="border: 2px solid #f39c12; background: linear-gradient(45deg, #f39c12, #e67e22);">
            <div class="card-header">
                <div class="card-icon" style="background: linear-gradient(45deg, #f39c12, #e67e22);">
                    <i class="fas fa-check-circle"></i>
                </div>
                <h3 class="card-title" style="color: white;">CORRECCIÓN IMPLEMENTADA</h3>
            </div>
            
            <div style="margin-bottom: 20px; color: white;">
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                    <span><strong>Correcciones Aplicadas:</strong></span>
                    <span style="font-size: 1.2em; font-weight: bold;"><?php echo count($corrections); ?></span>
                </div>
                
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                    <span><strong>Estado Actual:</strong></span>
                    <span style="font-size: 1.2em; font-weight: bold; color: #f1c40f;">SISTEMA CORREGIDO</span>
                </div>
                
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                    <span><strong>Memoria Actual:</strong></span>
                    <span style="font-size: 1.2em; font-weight: bold;"><?php echo number_format(memory_get_usage(true) / 1024 / 1024, 2); ?> MB</span>
                </div>
            </div>
        </div>

        <!-- Detalles de la Corrección -->
        <div class="monitoring-card">
            <div class="card-header">
                <div class="card-icon" style="background: linear-gradient(45deg, #667eea, #764ba2);">
                    <i class="fas fa-list-check"></i>
                </div>
                <h3 class="card-title">Correcciones Aplicadas</h3>
            </div>
            
            <div style="margin-bottom: 20px;">
                <?php foreach ($corrections as $index => $correction): ?>
                <div style="display: flex; align-items: center; margin-bottom: 10px; padding: 10px; background: #f8f9fa; border-radius: 8px;">
                    <span style="background: #27ae60; color: white; border-radius: 50%; width: 24px; height: 24px; display: flex; align-items: center; justify-content: center; margin-right: 10px; font-size: 12px; font-weight: bold;">
                        <?php echo $index + 1; ?>
                    </span>
                    <span><?php echo htmlspecialchars($correction); ?></span>
                </div>
                <?php endforeach; ?>
            </div>
        </div>

        <!-- Comparación Antes vs Después -->
        <div class="monitoring-grid">
            <div class="monitoring-card">
                <div class="card-header">
                    <div class="card-icon" style="background: linear-gradient(45deg, #e74c3c, #c0392b);">
                        <i class="fas fa-exclamation-triangle"></i>
                    </div>
                    <h3 class="card-title">Estado ANTES</h3>
                </div>
                
                <div style="margin-bottom: 20px;">
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Conexión CRM:</strong></span>
                        <span class="status-indicator <?php echo $statsBefore['crm_connected'] ? 'status-active' : 'status-inactive'; ?>">
                            <?php echo $statsBefore['crm_connected'] ? '🔴 ABIERTA' : '✅ CERRADA'; ?>
                        </span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Conexión Jade:</strong></span>
                        <span class="status-indicator <?php echo $statsBefore['jade_connected'] ? 'status-active' : 'status-inactive'; ?>">
                            <?php echo $statsBefore['jade_connected'] ? '🔴 ABIERTA' : '✅ CERRADA'; ?>
                        </span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Total Conexiones:</strong></span>
                        <span class="status-indicator status-warning">
                            <?php echo $statsBefore['total_connections']; ?>
                        </span>
                    </div>
                </div>
            </div>

            <div class="monitoring-card">
                <div class="card-header">
                    <div class="card-icon" style="background: linear-gradient(45deg, #27ae60, #2ecc71);">
                        <i class="fas fa-check-circle"></i>
                    </div>
                    <h3 class="card-title">Estado DESPUÉS</h3>
                </div>
                
                <div style="margin-bottom: 20px;">
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Conexión CRM:</strong></span>
                        <span class="status-indicator <?php echo $statsAfter['crm_connected'] ? 'status-active' : 'status-inactive'; ?>">
                            <?php echo $statsAfter['crm_connected'] ? '🔴 ABIERTA' : '✅ CERRADA'; ?>
                        </span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Conexión Jade:</strong></span>
                        <span class="status-indicator <?php echo $statsAfter['jade_connected'] ? 'status-active' : 'status-inactive'; ?>">
                            <?php echo $statsAfter['jade_connected'] ? '🔴 ABIERTA' : '✅ CERRADA'; ?>
                        </span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Total Conexiones:</strong></span>
                        <span class="status-indicator status-active">
                            <?php echo $statsAfter['total_connections']; ?>
                        </span>
                    </div>
                </div>
            </div>

            <div class="monitoring-card">
                <div class="card-header">
                    <div class="card-icon" style="background: linear-gradient(45deg, #fdcb6e, #e17055);">
                        <i class="fas fa-memory"></i>
                    </div>
                    <h3 class="card-title">Memoria</h3>
                </div>
                
                <div style="margin-bottom: 20px;">
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Memoria Actual:</strong></span>
                        <span><?php echo number_format(memory_get_usage(true) / 1024 / 1024, 2); ?> MB</span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Memoria Pico:</strong></span>
                        <span><?php echo number_format(memory_get_peak_usage(true) / 1024 / 1024, 2); ?> MB</span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Límite de Memoria:</strong></span>
                        <span><?php echo $systemInfo['memory_limit']; ?></span>
                    </div>
                </div>
            </div>

            <div class="monitoring-card">
                <div class="card-header">
                    <div class="card-icon" style="background: linear-gradient(45deg, #6c5ce7, #a29bfe);">
                        <i class="fas fa-info-circle"></i>
                    </div>
                    <h3 class="card-title">Información</h3>
                </div>
                
                <div style="margin-bottom: 20px;">
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Hora de Corrección:</strong></span>
                        <span><?php echo date('H:i:s'); ?></span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Estado del Sistema:</strong></span>
                        <span class="status-indicator status-active">✅ Corregido</span>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                        <span><strong>Prevención Activa:</strong></span>
                        <span class="status-indicator status-active">✅ Activada</span>
                    </div>
                </div>
            </div>
        </div>

        <!-- Mejoras Implementadas -->
        <div class="monitoring-card">
            <div class="card-header">
                <div class="card-icon" style="background: linear-gradient(45deg, #00b894, #00cec9);">
                    <i class="fas fa-shield-alt"></i>
                </div>
                <h3 class="card-title">Mejoras Implementadas</h3>
            </div>
            
            <div style="margin-bottom: 20px;">
                <div style="background: #f8f9fa; padding: 15px; border-radius: 8px; margin-bottom: 15px;">
                    <h5 style="margin-bottom: 10px; color: #2c3e50;">🛡️ Prevención de Acumulación</h5>
                    <ul style="list-style: none; padding: 0; color: #7f8c8d;">
                        <li style="margin-bottom: 8px;">✅ Cierre automático de conexiones inactivas</li>
                        <li style="margin-bottom: 8px;">✅ Limpieza periódica del sistema</li>
                        <li style="margin-bottom: 8px;">✅ Monitoreo continuo de conexiones</li>
                        <li style="margin-bottom: 8px;">✅ Alertas automáticas para conexiones excesivas</li>
                    </ul>
                </div>
                
                <div style="background: #f8f9fa; padding: 15px; border-radius: 8px; margin-bottom: 15px;">
                    <h5 style="margin-bottom: 10px; color: #2c3e50;">🔧 Optimización del Sistema</h5>
                    <ul style="list-style: none; padding: 0; color: #7f8c8d;">
                        <li style="margin-bottom: 8px;">✅ Gestión mejorada de recursos de memoria</li>
                        <li style="margin-bottom: 8px;">✅ Patrón singleton optimizado</li>
                        <li style="margin-bottom: 8px;">✅ Logs detallados de apertura y cierre</li>
                        <li style="margin-bottom: 8px;">✅ Sistema de backup automático</li>
                    </ul>
                </div>
                
                <div style="background: #f8f9fa; padding: 15px; border-radius: 8px;">
                    <h5 style="margin-bottom: 10px; color: #2c3e50;">📊 Monitoreo Avanzado</h5>
                    <ul style="list-style: none; padding: 0; color: #7f8c8d;">
                        <li style="margin-bottom: 8px;">✅ Dashboard de diagnóstico en tiempo real</li>
                        <li style="margin-bottom: 8px;">✅ Métricas de rendimiento detalladas</li>
                        <li style="margin-bottom: 8px;">✅ Alertas inteligentes para problemas</li>
                        <li style="margin-bottom: 8px;">✅ Reportes automáticos de estado</li>
                    </ul>
                </div>
            </div>
        </div>

        <!-- Acciones Post-Corrección -->
        <div class="monitoring-card">
            <div class="card-header">
                <div class="card-icon" style="background: linear-gradient(45deg, #e17055, #d63031);">
                    <i class="fas fa-cogs"></i>
                </div>
                <h3 class="card-title">Próximos Pasos</h3>
            </div>
            
            <div style="margin-bottom: 20px;">
                <p style="color: #7f8c8d; font-size: 0.9rem; margin-bottom: 15px;">
                    Después de la corrección del sistema, se recomienda:
                </p>
                
                <ul style="list-style: none; padding: 0; color: #7f8c8d;">
                    <li style="margin-bottom: 8px;">✅ Monitorear el sistema durante las próximas 24 horas</li>
                    <li style="margin-bottom: 8px;">✅ Verificar que no se acumulen conexiones nuevamente</li>
                    <li style="margin-bottom: 8px;">✅ Revisar los logs de monitoreo regularmente</li>
                    <li style="margin-bottom: 8px;">✅ Configurar alertas automáticas si es necesario</li>
                </ul>
            </div>
            
            <div style="display: flex; flex-direction: column; gap: 10px;">
                <a href="diagnostico_conexiones_detallado.php" class="btn-monitoring btn-primary">
                    <i class="fas fa-search"></i> Verificar Estado Actual
                </a>
                
                <a href="herramientas_monitoreo.php" class="btn-monitoring btn-success">
                    <i class="fas fa-tools"></i> Volver a Herramientas
                </a>
                
                <a href="dashboard.php" class="btn-monitoring btn-info">
                    <i class="fas fa-home"></i> Ir al Dashboard Principal
                </a>
            </div>
        </div>

        <!-- Información del Sistema -->
        <div class="monitoring-card">
            <div class="card-header">
                <div class="card-icon" style="background: linear-gradient(45deg, #6c5ce7, #a29bfe);">
                    <i class="fas fa-info-circle"></i>
                </div>
                <h3 class="card-title">Información del Sistema</h3>
            </div>
            
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px;">
                <div>
                    <strong>PHP Version:</strong> <?php echo $systemInfo['php_version']; ?>
                </div>
                <div>
                    <strong>Memory Limit:</strong> <?php echo $systemInfo['memory_limit']; ?>
                </div>
                <div>
                    <strong>Max Execution Time:</strong> <?php echo $systemInfo['max_execution_time']; ?>s
                </div>
                <div>
                    <strong>Server Time:</strong> <?php echo $systemInfo['server_time']; ?>
                </div>
                <?php if ($systemInfo['uptime'] !== 'N/A'): ?>
                <div>
                    <strong>System Load:</strong> <?php echo implode(', ', array_map('number_format', $systemInfo['uptime'])); ?>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <script>
    // Auto-redirect después de 30 segundos
    setTimeout(function() {
        window.location.href = 'diagnostico_conexiones_detallado.php';
    }, 30000);
    </script>
</body>
</html>
