<?php
/**
 * Sistema Avanzado de Gestión de Conexiones
 * Incluye pool de conexiones, logging, métricas y timeouts configurables
 */

class AdvancedDatabaseManager {
    private static $instance = null;
    private $crmConnection = null;
    private $jadeConnection = null;
    private $connectionCount = 0;
    private $connectionLog = [];
    private $performanceMetrics = [];
    private $maxConnections = 10;
    private $connectionTimeout = 30; // segundos
    private $lastActivity = [];
    
    private function __construct() {
        $this->initializeLogging();
    }
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Inicializar sistema de logging
     */
    private function initializeLogging() {
        if (!is_dir('logs')) {
            mkdir('logs', 0755, true);
        }
    }
    
    /**
     * Obtener conexión CRM con pool y timeout
     */
    public function getCRMConnection() {
        $startTime = microtime(true);
        
        if ($this->crmConnection === null) {
            try {
                $this->logConnection('CRM', 'OPENING', 'Iniciando conexión CRM');
                
                $this->crmConnection = new PDO(
                    "mysql:host=localhost;dbname=innovadi_crm_panel",
                    "innovadi_jade",
                    "gY@HlRv7nVSL9*ry",
                    array(
                        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                        PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8",
                        PDO::ATTR_TIMEOUT => $this->connectionTimeout
                    )
                );
                
                $this->connectionCount++;
                $this->lastActivity['crm'] = time();
                
                $this->logConnection('CRM', 'SUCCESS', 'Conexión CRM establecida');
                $this->recordPerformance('CRM', 'connection_time', microtime(true) - $startTime);
                
            } catch(PDOException $e) {
                $this->logConnection('CRM', 'ERROR', 'Error de conexión CRM: ' . $e->getMessage());
                throw new Exception("Error de conexión a la base de datos CRM");
            }
        } else {
            $this->lastActivity['crm'] = time();
            $this->logConnection('CRM', 'REUSE', 'Reutilizando conexión CRM existente');
        }
        
        return $this->crmConnection;
    }
    
    /**
     * Obtener conexión Jade con pool y timeout
     */
    public function getJadeConnection() {
        $startTime = microtime(true);
        
        if ($this->jadeConnection === null) {
            try {
                $this->logConnection('JADE', 'OPENING', 'Iniciando conexión Jade');
                
                $this->jadeConnection = new PDO(
                    "mysql:host=localhost;dbname=innovadi_jade",
                    "innovadi_jade",
                    "gY@HlRv7nVSL9*ry",
                    array(
                        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                        PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8",
                        PDO::ATTR_TIMEOUT => $this->connectionTimeout
                    )
                );
                
                $this->connectionCount++;
                $this->lastActivity['jade'] = time();
                
                $this->logConnection('JADE', 'SUCCESS', 'Conexión Jade establecida');
                $this->recordPerformance('JADE', 'connection_time', microtime(true) - $startTime);
                
            } catch(PDOException $e) {
                $this->logConnection('JADE', 'ERROR', 'Error de conexión Jade: ' . $e->getMessage());
                throw new Exception("Error de conexión a la base de datos Jade");
            }
        } else {
            $this->lastActivity['jade'] = time();
            $this->logConnection('JADE', 'REUSE', 'Reutilizando conexión Jade existente');
        }
        
        return $this->jadeConnection;
    }
    
    /**
     * Cerrar conexiones con logging
     */
    public function closeConnections() {
        if ($this->crmConnection !== null) {
            $this->logConnection('CRM', 'CLOSING', 'Cerrando conexión CRM');
            $this->crmConnection = null;
            $this->connectionCount--;
        }
        if ($this->jadeConnection !== null) {
            $this->logConnection('JADE', 'CLOSING', 'Cerrando conexión Jade');
            $this->jadeConnection = null;
            $this->connectionCount--;
        }
    }
    
    /**
     * Destructor para asegurar cierre de conexiones
     */
    public function __destruct() {
        $this->closeConnections();
    }
    
    /**
     * Logging de conexiones
     */
    private function logConnection($type, $action, $message) {
        $logEntry = [
            'timestamp' => date('Y-m-d H:i:s'),
            'type' => $type,
            'action' => $action,
            'message' => $message,
            'memory_usage' => memory_get_usage(true),
            'peak_memory' => memory_get_peak_usage(true)
        ];
        
        $this->connectionLog[] = $logEntry;
        
        // Escribir a archivo de log
        $logFile = 'logs/database_' . date('Y-m-d') . '.log';
        $logLine = json_encode($logEntry) . "\n";
        file_put_contents($logFile, $logLine, FILE_APPEND | LOCK_EX);
    }
    
    /**
     * Registrar métricas de rendimiento
     */
    private function recordPerformance($type, $metric, $value) {
        if (!isset($this->performanceMetrics[$type])) {
            $this->performanceMetrics[$type] = [];
        }
        if (!isset($this->performanceMetrics[$type][$metric])) {
            $this->performanceMetrics[$type][$metric] = [];
        }
        
        $this->performanceMetrics[$type][$metric][] = $value;
        
        // Mantener solo las últimas 100 métricas
        if (count($this->performanceMetrics[$type][$metric]) > 100) {
            array_shift($this->performanceMetrics[$type][$metric]);
        }
    }
    
    /**
     * Obtener estadísticas avanzadas
     */
    public function getAdvancedStats() {
        $stats = [
            'crm_connected' => $this->crmConnection !== null,
            'jade_connected' => $this->jadeConnection !== null,
            'total_connections' => $this->connectionCount,
            'max_connections' => $this->maxConnections,
            'connection_timeout' => $this->connectionTimeout,
            'last_activity' => $this->lastActivity,
            'performance_metrics' => $this->performanceMetrics,
            'recent_logs' => array_slice($this->connectionLog, -10)
        ];
        
        // Calcular métricas de rendimiento
        foreach ($this->performanceMetrics as $type => $metrics) {
            foreach ($metrics as $metric => $values) {
                if (!empty($values)) {
                    $stats['performance_summary'][$type][$metric] = [
                        'avg' => array_sum($values) / count($values),
                        'min' => min($values),
                        'max' => max($values),
                        'count' => count($values)
                    ];
                }
            }
        }
        
        return $stats;
    }
    
    /**
     * Verificar health de las conexiones
     */
    public function checkConnectionHealth() {
        $health = [
            'crm' => ['status' => 'unknown', 'last_activity' => null],
            'jade' => ['status' => 'unknown', 'last_activity' => null]
        ];
        
        if ($this->crmConnection !== null) {
            try {
                $this->crmConnection->query("SELECT 1");
                $health['crm']['status'] = 'healthy';
                $health['crm']['last_activity'] = $this->lastActivity['crm'] ?? null;
            } catch (Exception $e) {
                $health['crm']['status'] = 'unhealthy';
                $health['crm']['error'] = $e->getMessage();
            }
        }
        
        if ($this->jadeConnection !== null) {
            try {
                $this->jadeConnection->query("SELECT 1");
                $health['jade']['status'] = 'healthy';
                $health['jade']['last_activity'] = $this->lastActivity['jade'] ?? null;
            } catch (Exception $e) {
                $health['jade']['status'] = 'unhealthy';
                $health['jade']['error'] = $e->getMessage();
            }
        }
        
        return $health;
    }
    
    /**
     * Limpiar conexiones inactivas
     */
    public function cleanupInactiveConnections($timeout = 300) {
        $currentTime = time();
        $cleaned = 0;
        
        if ($this->crmConnection !== null && 
            isset($this->lastActivity['crm']) && 
            ($currentTime - $this->lastActivity['crm']) > $timeout) {
            $this->logConnection('CRM', 'CLEANUP', 'Cerrando conexión CRM inactiva');
            $this->crmConnection = null;
            $this->connectionCount--;
            $cleaned++;
        }
        
        if ($this->jadeConnection !== null && 
            isset($this->lastActivity['jade']) && 
            ($currentTime - $this->lastActivity['jade']) > $timeout) {
            $this->logConnection('JADE', 'CLEANUP', 'Cerrando conexión Jade inactiva');
            $this->jadeConnection = null;
            $this->connectionCount--;
            $cleaned++;
        }
        
        return $cleaned;
    }
}

// Funciones helper avanzadas
function getAdvancedDB($type = 'crm') {
    $manager = AdvancedDatabaseManager::getInstance();
    switch(strtolower($type)) {
        case 'jade':
            return $manager->getJadeConnection();
        case 'crm':
        default:
            return $manager->getCRMConnection();
    }
}

function closeAdvancedDB() {
    $manager = AdvancedDatabaseManager::getInstance();
    $manager->closeConnections();
}

function getAdvancedDBStats() {
    $manager = AdvancedDatabaseManager::getInstance();
    return $manager->getAdvancedStats();
}

function checkDBHealth() {
    $manager = AdvancedDatabaseManager::getInstance();
    return $manager->checkConnectionHealth();
}

function cleanupDBConnections($timeout = 300) {
    $manager = AdvancedDatabaseManager::getInstance();
    return $manager->cleanupInactiveConnections($timeout);
}

// Función para obtener logs de conexiones
function getDBLogs($lines = 50) {
    $logFile = 'logs/database_' . date('Y-m-d') . '.log';
    if (file_exists($logFile)) {
        $logs = file($logFile);
        return array_slice($logs, -$lines);
    }
    return [];
}
?>
