<?php
session_start();
if(!isset($_SESSION['user_id'])) {
    header("Location: index.php");
    exit();
}

require_once "config/database_singleton.php";

// Configuración de paginación
$registros_por_pagina = 25;
$pagina = isset($_GET['pagina']) ? (int)$_GET['pagina'] : 1;
$inicio = ($pagina - 1) * $registros_por_pagina;

// Conexión a la base de datos Jade
$db = getDB('jade');

// Obtener término de búsqueda
$busqueda = isset($_GET['buscar']) ? trim($_GET['buscar']) : '';

try {
    // Preparar la cláusula WHERE para la búsqueda
    $where_clause = "";
    $params = [];
    
    if (!empty($busqueda)) {
        $where_clause = " WHERE (c.char_name LIKE :busqueda OR c.account_name LIKE :busqueda)";
        $params[':busqueda'] = "%{$busqueda}%";
    }

    // Obtener total de registros con el filtro
    $query_total = "SELECT COUNT(*) as total FROM characters c" . $where_clause;
    $stmt_total = $db->prepare($query_total);
    if (!empty($params)) {
        $stmt_total->execute($params);
    } else {
        $stmt_total->execute();
    }
    $total_registros = $stmt_total->fetch(PDO::FETCH_ASSOC)['total'];
    $total_paginas = ceil($total_registros / $registros_por_pagina);

    // Ajustar página actual si está fuera de rango
    if ($pagina > $total_paginas && $total_paginas > 0) {
        $pagina = 1;
        $inicio = 0;
    }

    // Obtener personajes con su cuenta asociada (consulta optimizada)
    $query = "SELECT c.obj_Id, c.char_name, c.account_name as login 
              FROM characters c" . $where_clause . " 
              ORDER BY c.char_name 
              LIMIT :inicio, :registros_por_pagina";
    
    $stmt = $db->prepare($query);
    if (!empty($params)) {
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
    }
    $stmt->bindValue(':inicio', $inicio, PDO::PARAM_INT);
    $stmt->bindValue(':registros_por_pagina', $registros_por_pagina, PDO::PARAM_INT);
    $stmt->execute();
    $characters = $stmt->fetchAll(PDO::FETCH_ASSOC);

} catch(PDOException $e) {
    $error = "Error al obtener datos: " . $e->getMessage();
}

// Definir los nombres de columnas en español
$column_names = [
    'obj_Id' => 'ID',
    'char_name' => 'Nombre del Personaje',
    'login' => 'Cuenta'
];
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>CRM - Personajes</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="assets/css/dark-theme.css" rel="stylesheet">

</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-2 sidebar">
                <h3 class="text-white text-center mb-4">CRM Panel</h3>
                <a href="dashboard.php"><i class="fas fa-home"></i> Dashboard</a>
                <a href="accounts.php"><i class="fas fa-user-circle"></i> Accounts</a>
                <a href="characters.php" class="active"><i class="fas fa-gamepad"></i> Personajes</a>
                <a href="geolocalizacion.php"><i class="fas fa-globe"></i> Geolocalización</a>
                <a href="logout.php"><i class="fas fa-sign-out-alt"></i> Cerrar Sesión</a>
            </div>

            <!-- Main Content -->
            <div class="col-md-10 main-content">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <div>
                        <h2>Personajes</h2>
                        <p class="text-muted mb-0">
                            Total de personajes: <strong><?php echo number_format($total_registros, 0, ',', '.'); ?></strong>
                        </p>
                    </div>
                    <div class="search-box">
                        <form class="search-form" method="GET">
                            <div class="input-group">
                                <input type="text" name="buscar" class="form-control" placeholder="Buscar personaje..." value="<?php echo htmlspecialchars($_GET['buscar'] ?? ''); ?>">
                                <button type="submit" class="btn btn-primary">Buscar</button>
                            </div>
                        </form>
                    </div>
                </div>

                <?php if(isset($error)): ?>
                    <div class="alert alert-danger"><?php echo $error; ?></div>
                <?php endif; ?>

                <div class="card">
                    <div class="card-body">
                        <!-- Loading indicator -->
                        <div id="loadingIndicator" class="text-center py-4" style="display: none;">
                            <div class="spinner-border text-primary" role="status">
                                <span class="visually-hidden">Cargando...</span>
                            </div>
                            <p class="mt-2 text-muted">Cargando personajes...</p>
                        </div>

                        <div class="table-responsive">
                            <table class="table table-dark table-striped table-hover">
                                <thead>
                                    <tr>
                                        <?php foreach($column_names as $name): ?>
                                            <th><?php echo htmlspecialchars($name); ?></th>
                                        <?php endforeach; ?>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach($characters as $character): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($character['obj_Id']); ?></td>
                                            <td><?php echo htmlspecialchars($character['char_name']); ?></td>
                                            <td><?php echo htmlspecialchars($character['login']); ?></td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>

                        <!-- Paginación -->
                        <?php if($total_paginas > 1): ?>
                            <div class="d-flex justify-content-center mt-4">
                                <nav aria-label="Page navigation">
                                    <ul class="pagination mb-0">
                                        <?php if($pagina > 1): ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?pagina=<?php echo $pagina-1; ?><?php echo !empty($busqueda) ? '&buscar='.urlencode($busqueda) : ''; ?>" aria-label="Previous">
                                                    <span aria-hidden="true">&laquo;</span>
                                                </a>
                                            </li>
                                        <?php endif; ?>

                                        <?php
                                        $rango = 2;
                                        $inicio_rango = max(1, $pagina - $rango);
                                        $fin_rango = min($total_paginas, $pagina + $rango);

                                        if ($inicio_rango > 1) {
                                            echo '<li class="page-item"><a class="page-link" href="?pagina=1">1</a></li>';
                                            if ($inicio_rango > 2) {
                                                echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                                            }
                                        }

                                        for($i = $inicio_rango; $i <= $fin_rango; $i++): ?>
                                            <li class="page-item <?php echo $pagina == $i ? 'active' : ''; ?>">
                                                <a class="page-link" href="?pagina=<?php echo $i; ?><?php echo !empty($busqueda) ? '&buscar='.urlencode($busqueda) : ''; ?>"><?php echo $i; ?></a>
                                            </li>
                                        <?php endfor;

                                        if ($fin_rango < $total_paginas) {
                                            if ($fin_rango < $total_paginas - 1) {
                                                echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                                            }
                                            echo '<li class="page-item"><a class="page-link" href="?pagina=' . $total_paginas . '">' . $total_paginas . '</a></li>';
                                        }
                                        ?>

                                        <?php if($pagina < $total_paginas): ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?pagina=<?php echo $pagina+1; ?><?php echo !empty($busqueda) ? '&buscar='.urlencode($busqueda) : ''; ?>" aria-label="Next">
                                                    <span aria-hidden="true">&raquo;</span>
                                                </a>
                                            </li>
                                        <?php endif; ?>
                                    </ul>
                                </nav>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        const loadingIndicator = document.getElementById('loadingIndicator');
        const tableContent = document.querySelector('.table-responsive');
        
        // Mostrar indicador de carga
        loadingIndicator.style.display = 'block';
        tableContent.style.opacity = '0';
        
        // Ocultar indicador cuando la página termine de cargar
        window.addEventListener('load', function() {
            setTimeout(function() {
                loadingIndicator.style.display = 'none';
                tableContent.style.opacity = '1';
                tableContent.style.transition = 'opacity 0.3s ease';
            }, 500);
        });

        // Manejar clics en la paginación
        document.querySelectorAll('.pagination a').forEach(link => {
            link.addEventListener('click', function() {
                loadingIndicator.style.display = 'block';
                tableContent.style.opacity = '0';
            });
        });
    });
    </script>
</body>
</html>
