<?php
session_start();
if(!isset($_SESSION['user_id'])) {
    header("Location: index.php");
    exit();
}

require_once "config/database_singleton.php";
require_once "includes/password_utils.php";

// Configuración de paginación y búsqueda
$registros_por_pagina = 25;
$pagina = isset($_GET['pagina']) ? (int)$_GET['pagina'] : 1;
$busqueda = isset($_GET['buscar']) ? trim($_GET['buscar']) : '';
$inicio = ($pagina - 1) * $registros_por_pagina;

// Conexión a la base de datos Jade
$db = getDB('jade');

try {
    // Construir la consulta base
    $where_clause = "";
    $params = [];
    
    if (!empty($busqueda)) {
        $where_clause = " WHERE login LIKE :busqueda 
                         OR correo LIKE :busqueda 
                         OR last_active LIKE :busqueda 
                         OR lastip LIKE :busqueda
                         OR password LIKE :busqueda";
        $params[':busqueda'] = "%{$busqueda}%";
    }

    // Obtener total de registros únicos con el filtro
    $query_total = "SELECT COUNT(DISTINCT login) as total FROM accounts" . $where_clause;
    $stmt_total = $db->prepare($query_total);
    if (!empty($params)) {
        $stmt_total->execute($params);
    } else {
        $stmt_total->execute();
    }
    $total_registros = $stmt_total->fetch(PDO::FETCH_ASSOC)['total'];
    $total_paginas = ceil($total_registros / $registros_por_pagina);

    // Ajustar página actual si está fuera de rango
    if ($pagina > $total_paginas) {
        $pagina = 1;
        $inicio = 0;
    }

    // Obtener registros únicos filtrados y paginados
    $query = "SELECT login, correo, last_active, lastip, password
              FROM accounts" . $where_clause . "
              GROUP BY login 
              ORDER BY last_active DESC 
              LIMIT :inicio, :registros_por_pagina";
    
    $stmt = $db->prepare($query);
    if (!empty($params)) {
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
    }
    $stmt->bindValue(':inicio', $inicio, PDO::PARAM_INT);
    $stmt->bindValue(':registros_por_pagina', $registros_por_pagina, PDO::PARAM_INT);
    $stmt->execute();
    $accounts = $stmt->fetchAll(PDO::FETCH_ASSOC);

} catch(PDOException $e) {
    $error = "Error al obtener datos: " . $e->getMessage();
}

// Definir los nombres de columnas en español
$column_names = [
    'login' => 'Usuario',
    'correo' => 'Correo',
    'last_active' => 'Última Actividad',
    'lastip' => 'Última IP',
    'password' => 'Contraseña'
];
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>CRM - Accounts</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="assets/css/dark-theme.css" rel="stylesheet">

</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-2 sidebar">
                <h3 class="text-white text-center mb-4">CRM Panel</h3>
                <a href="dashboard.php"><i class="fas fa-home"></i> Dashboard</a>
                <a href="accounts.php" class="active"><i class="fas fa-user-circle"></i> Accounts</a>
                <a href="characters.php"><i class="fas fa-gamepad"></i> Personajes</a>
                <a href="geolocalizacion.php"><i class="fas fa-globe"></i> Geolocalización</a>
                <a href="logout.php"><i class="fas fa-sign-out-alt"></i> Cerrar Sesión</a>
            </div>

            <!-- Main Content -->
            <div class="col-md-10 main-content">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <div>
                        <h2>Accounts</h2>
                        <p class="text-muted mb-0">
                            Total de cuentas: <strong><?php echo number_format($total_registros, 0, ',', '.'); ?></strong>
                            <?php if(!empty($busqueda)): ?>
                                <br>Resultados encontrados: <strong><?php echo number_format($total_registros, 0, ',', '.'); ?></strong>
                            <?php endif; ?>
                        </p>
                    </div>
                    <div class="search-box">
                        <form class="search-form" method="GET">
                            <input type="text" name="buscar" class="form-control" placeholder="Buscar..." value="<?php echo htmlspecialchars($busqueda); ?>">
                            <button type="submit" class="btn btn-primary">Buscar</button>
                            <?php if(!empty($busqueda)): ?>
                                <a href="accounts.php" class="btn btn-secondary">Limpiar</a>
                            <?php endif; ?>
                        </form>
                    </div>
                </div>
                
                <?php if(isset($error)): ?>
                    <div class="alert alert-danger"><?php echo $error; ?></div>
                <?php endif; ?>

                <div class="card">
                    <div class="card-body">
                        <?php if($total_registros > 0): ?>
                            <div class="table-responsive">
                                <table class="table table-dark table-striped table-hover">
                                    <thead>
                                        <tr>
                                            <?php foreach($column_names as $column => $name): ?>
                                                <th><?php echo htmlspecialchars($name); ?></th>
                                            <?php endforeach; ?>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach($accounts as $account): ?>
                                            <tr>
                                                                                            <td><?php echo htmlspecialchars($account['login']); ?></td>
                                            <td><?php echo htmlspecialchars($account['correo']); ?></td>
                                            <td><?php echo htmlspecialchars($account['last_active']); ?></td>
                                            <td><?php echo htmlspecialchars($account['lastip']); ?></td>
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <div class="password-display">
                                                        <?php 
                                                        $decrypted = decryptPasswordAttempt($account['password']);
                                                        if ($decrypted !== false) {
                                                            echo '<span class="text-success"><i class="fas fa-unlock"></i> ' . htmlspecialchars($decrypted) . '</span>';
                                                        } else {
                                                            echo '<code>' . htmlspecialchars($account['password']) . '</code>';
                                                        }
                                                        ?>
                                                    </div>
                                                    <button class="btn btn-sm btn-outline-primary ms-2 edit-password" 
                                                            data-login="<?php echo htmlspecialchars($account['login']); ?>"
                                                            title="Editar contraseña">
                                                        <i class="fas fa-edit"></i>
                                                    </button>
                                                </div>
                                            </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>

                            <!-- Paginación -->
                            <?php if($total_paginas > 1): ?>
                                <div class="d-flex justify-content-center mt-4">
                                    <nav aria-label="Page navigation">
                                        <ul class="pagination mb-0">
                                            <?php if($pagina > 1): ?>
                                                <li class="page-item">
                                                    <a class="page-link" href="?pagina=<?php echo $pagina-1; ?><?php echo !empty($busqueda) ? '&buscar='.urlencode($busqueda) : ''; ?>" aria-label="Previous">
                                                        <span aria-hidden="true">&laquo;</span>
                                                    </a>
                                                </li>
                                            <?php endif; ?>

                                            <?php
                                            $rango = 2;
                                            $inicio_rango = max(1, $pagina - $rango);
                                            $fin_rango = min($total_paginas, $pagina + $rango);

                                            if ($inicio_rango > 1) {
                                                echo '<li class="page-item"><a class="page-link" href="?pagina=1'.(!empty($busqueda) ? '&buscar='.urlencode($busqueda) : '').'">1</a></li>';
                                                if ($inicio_rango > 2) {
                                                    echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                                                }
                                            }

                                            for($i = $inicio_rango; $i <= $fin_rango; $i++): ?>
                                                <li class="page-item <?php echo $pagina == $i ? 'active' : ''; ?>">
                                                    <a class="page-link" href="?pagina=<?php echo $i; ?><?php echo !empty($busqueda) ? '&buscar='.urlencode($busqueda) : ''; ?>"><?php echo $i; ?></a>
                                                </li>
                                            <?php endfor;

                                            if ($fin_rango < $total_paginas) {
                                                if ($fin_rango < $total_paginas - 1) {
                                                    echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                                                }
                                                echo '<li class="page-item"><a class="page-link" href="?pagina='.$total_paginas.(!empty($busqueda) ? '&buscar='.urlencode($busqueda) : '').'">'.$total_paginas.'</a></li>';
                                            }
                                            ?>

                                            <?php if($pagina < $total_paginas): ?>
                                                <li class="page-item">
                                                    <a class="page-link" href="?pagina=<?php echo $pagina+1; ?><?php echo !empty($busqueda) ? '&buscar='.urlencode($busqueda) : ''; ?>" aria-label="Next">
                                                        <span aria-hidden="true">&raquo;</span>
                                                    </a>
                                                </li>
                                            <?php endif; ?>
                                        </ul>
                                    </nav>
                                </div>
                            <?php endif; ?>
                        <?php else: ?>
                            <div class="alert alert-info">No se encontraron registros<?php echo !empty($busqueda) ? ' para la búsqueda: "'.htmlspecialchars($busqueda).'"' : ''; ?></div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Modal para editar contraseña -->
    <div class="modal fade" id="editPasswordModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Editar Contraseña</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <input type="hidden" id="editLoginInput">
                    <div class="mb-3">
                        <label for="newPasswordInput" class="form-label">Nueva Contraseña</label>
                        <input type="text" class="form-control" id="newPasswordInput" required>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="button" class="btn btn-primary" id="savePasswordBtn">Guardar</button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        const modal = new bootstrap.Modal(document.getElementById('editPasswordModal'));
        const editLoginInput = document.getElementById('editLoginInput');
        const newPasswordInput = document.getElementById('newPasswordInput');
        const savePasswordBtn = document.getElementById('savePasswordBtn');

        // Manejar clic en botón de editar
        document.querySelectorAll('.edit-password').forEach(button => {
            button.addEventListener('click', function() {
                const login = this.dataset.login;
                editLoginInput.value = login;
                newPasswordInput.value = '';
                modal.show();
            });
        });

        // Manejar guardado de contraseña
        savePasswordBtn.addEventListener('click', async function() {
            const login = editLoginInput.value;
            const password = newPasswordInput.value;

            if (!password) {
                alert('Por favor ingrese una contraseña');
                return;
            }

            try {
                const response = await fetch('update_password.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({ login, password })
                });

                const data = await response.json();

                if (data.success) {
                    alert('Contraseña actualizada correctamente');
                    modal.hide();
                    location.reload(); // Recargar para ver los cambios
                } else {
                    alert('Error: ' + data.message);
                }
            } catch (error) {
                alert('Error al actualizar la contraseña');
                console.error(error);
            }
        });
    });
    </script>
</body>
</html>